#!/usr/bin/env python3
"""
NanoCERN Knowledge Reactor - CLI Interface
Constraint-based reasoning engine using Knowledge Units (KUs).
"""

import argparse
import json
import sys
from pathlib import Path

def load_ku(ku_file: Path) -> dict:
    """Load a Knowledge Unit from file."""
    with open(ku_file, 'r') as f:
        return json.load(f)

def validate_ku(ku: dict) -> bool:
    """Validate KU structure."""
    required_fields = ['id', 'domain', 'invariant', 'applies_if', 'failure_modes']
    return all(field in ku for field in required_fields)

def list_kus(atoms_dir: Path):
    """List all available Knowledge Units."""
    ku_files = list(atoms_dir.glob('*.ku'))
    print(f"\n{'='*80}")
    print(f"KNOWLEDGE UNITS ({len(ku_files)} total)")
    print(f"{'='*80}\n")
    
    for ku_file in sorted(ku_files):
        ku = load_ku(ku_file)
        print(f"[{ku['domain'].upper()}] {ku['id']}")
        print(f"  Invariant: {ku['invariant'][:70]}...")
        print()

def check_constraint(ku_file: Path, state: dict):
    """Check if a state satisfies KU constraints."""
    ku = load_ku(ku_file)
    
    print(f"\n{'='*80}")
    print(f"CONSTRAINT CHECK: {ku['id']}")
    print(f"{'='*80}\n")
    print(f"Invariant: {ku['invariant']}\n")
    
    # Check applicability conditions
    violations = []
    for condition, threshold in ku['applies_if'].items():
        if condition in state:
            actual = state[condition]
            print(f"  {condition}: {actual} (expected: {threshold})")
            
            # Simple threshold check (extend for complex logic)
            if isinstance(threshold, str) and threshold.startswith('>'):
                expected_val = float(threshold[1:])
                if actual <= expected_val:
                    violations.append(condition)
            elif isinstance(threshold, str) and threshold.startswith('<'):
                expected_val = float(threshold[1:])
                if actual >= expected_val:
                    violations.append(condition)
        else:
            print(f"  {condition}: NOT PROVIDED")
            violations.append(condition)
    
    print()
    if violations:
        print(f"❌ CONSTRAINT VIOLATED")
        print(f"   Failed conditions: {', '.join(violations)}")
        print(f"   Failure modes: {', '.join(ku['failure_modes'])}")
    else:
        print(f"✅ CONSTRAINT SATISFIED")
    print()

def main():
    parser = argparse.ArgumentParser(
        description='NanoCERN Knowledge Reactor - Constraint-based reasoning',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # List all Knowledge Units
  nanocern list
  
  # Check constraint against state
  nanocern check MED-METFORMIN-RENAL.ku --state '{"GFR": 25}'
  
  # Validate KU structure
  nanocern validate atoms/MED-METFORMIN-RENAL.ku
        """
    )
    
    parser.add_argument('command', choices=['list', 'check', 'validate'],
                       help='Command to execute')
    parser.add_argument('ku_file', nargs='?', help='Knowledge Unit file')
    parser.add_argument('--state', type=str, help='JSON state to check')
    parser.add_argument('--atoms-dir', type=Path, default=Path('atoms'),
                       help='Directory containing KU files')
    
    args = parser.parse_args()
    
    if args.command == 'list':
        list_kus(args.atoms_dir)
    
    elif args.command == 'validate':
        if not args.ku_file:
            print("Error: ku_file required for validate command")
            sys.exit(1)
        ku = load_ku(Path(args.ku_file))
        if validate_ku(ku):
            print(f"✅ {args.ku_file} is valid")
        else:
            print(f"❌ {args.ku_file} is invalid")
            sys.exit(1)
    
    elif args.command == 'check':
        if not args.ku_file or not args.state:
            print("Error: ku_file and --state required for check command")
            sys.exit(1)
        state = json.loads(args.state)
        check_constraint(Path(args.ku_file), state)

if __name__ == '__main__':
    main()
