#!/usr/bin/env python3
"""
KU-Refinery: JSON to KUAT Binary Batch Processor
Converts the NanoWiki extraction output into high-performance binary atoms.
"""

import os
import json
from kuat_binary import serialize_ku_to_kuat

def run_refinery(input_dir: str, output_dir: str):
    """
    Scans input_dir for JSON KUs and serializes them to output_dir as .ku files.
    """
    print("="*60)
    print(f"KU-REFINERY: Processing {input_dir}")
    print("="*60)
    
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)
        
    physics_dir = os.path.join(input_dir, "physics")
    if not os.path.exists(physics_dir):
        print("✗ No physics KUs found to refine.")
        return

    json_files = [f for f in os.listdir(physics_dir) if f.endswith(".json")]
    print(f"✓ Found {len(json_files)} JSON atoms. Commencing refinement...")
    
    count = 0
    for filename in json_files:
        try:
            with open(os.path.join(physics_dir, filename), 'r') as f:
                ku_dict = json.load(f)
            
            # Ensure required fields for KUAT
            if "type" not in ku_dict:
                ku_dict["type"] = "A" # Default to Algebraic for physics
            
            # Map complex NanoWiki structure to flat applies_if for the Minimal Reactor
            # This is a temporary bridge until Phase III.
            if isinstance(ku_dict["applies_if"], dict) and "condition1" in ku_dict["applies_if"]:
                 # Create a dummy numerical envelope if none exists to allow rejection tests
                 ku_dict["applies_if"] = {"Value >": 0}
            
            binary_data = serialize_ku_to_kuat(ku_dict)
            
            output_filename = filename.replace(".json", ".ku")
            with open(os.path.join(output_dir, output_filename), 'wb') as f:
                f.write(binary_data)
            
            count += 1
            if count % 50 == 0:
                print(f"  Processed {count}/{len(json_files)}...")
                
        except Exception as e:
            print(f"  ✗ Failed to refine {filename}: {e}")

    print("\n" + "="*60)
    print(f"REFINERY COMPLETE: {count} atoms refined to .ku format.")
    print(f"Output: {output_dir}")
    print("="*60)

if __name__ == "__main__":
    run_refinery("../extracted_kus", "atoms")
