#!/usr/bin/env python3
"""
Homeostasis Measurement Engine
Tracks balance across 8 fundamental physiological axes.
"""

from dataclasses import dataclass
from typing import Dict, List, Tuple
from enum import Enum


class AxisState(Enum):
    """Homeostatic axis balance state."""
    BALANCED = "BALANCED"
    SUBOPTIMAL = "SUBOPTIMAL"
    IMBALANCED = "IMBALANCED"
    CRITICAL = "CRITICAL"


@dataclass
class HomeostasisAxis:
    """Base class for homeostatic axes."""
    name: str
    score: float = 100.0  # 0-100
    state: AxisState = AxisState.BALANCED
    
    def classify_state(self) -> AxisState:
        """Classify axis state based on score."""
        if self.score >= 85:
            return AxisState.BALANCED
        elif self.score >= 70:
            return AxisState.SUBOPTIMAL
        elif self.score >= 50:
            return AxisState.IMBALANCED
        else:
            return AxisState.CRITICAL


class PHBalanceAxis(HomeostasisAxis):
    """pH Balance (Acid-Base Homeostasis)."""
    
    def __init__(self):
        super().__init__(name="pH Balance")
        self.blood_ph = 7.40
        self.bicarbonate_meq_l = 24
        self.pco2_mmhg = 40
        self.anion_gap = 12
    
    def calculate_score(self) -> float:
        """Calculate pH balance score."""
        scores = []
        
        # Blood pH (7.35-7.45)
        if 7.35 <= self.blood_ph <= 7.45:
            scores.append(100)
        elif 7.30 <= self.blood_ph < 7.35 or 7.45 < self.blood_ph <= 7.50:
            scores.append(70)
        else:
            scores.append(30)
        
        # Bicarbonate (22-28)
        if 22 <= self.bicarbonate_meq_l <= 28:
            scores.append(100)
        else:
            deviation = abs(self.bicarbonate_meq_l - 25) / 25 * 100
            scores.append(max(0, 100 - deviation))
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class RedoxBalanceAxis(HomeostasisAxis):
    """Redox Balance (Oxidative Stress)."""
    
    def __init__(self):
        super().__init__(name="Redox Balance")
        self.glutathione_ratio = 10  # GSH:GSSG
        self.lipid_peroxidation = 3
        self.mitochondrial_function_percent = 100
    
    def calculate_score(self) -> float:
        """Calculate redox balance score."""
        scores = []
        
        # Glutathione ratio (>10:1 is optimal)
        if self.glutathione_ratio >= 10:
            scores.append(100)
        else:
            scores.append((self.glutathione_ratio / 10) * 100)
        
        # Lipid peroxidation (<5 is optimal)
        if self.lipid_peroxidation < 5:
            scores.append(100)
        else:
            scores.append(max(0, 100 - (self.lipid_peroxidation - 5) * 10))
        
        # Mitochondrial function
        scores.append(self.mitochondrial_function_percent)
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class InflammatoryBalanceAxis(HomeostasisAxis):
    """Inflammatory Balance (Pro/Anti-inflammatory)."""
    
    def __init__(self):
        super().__init__(name="Inflammatory Balance")
        self.il6_pg_ml = 3
        self.il10_pg_ml = 12
        self.tnf_alpha_pg_ml = 15
        self.crp_mg_l = 2
    
    def calculate_score(self) -> float:
        """Calculate inflammatory balance score."""
        scores = []
        
        # IL-6 (<5 is optimal)
        if self.il6_pg_ml < 5:
            scores.append(100)
        else:
            scores.append(max(0, 100 - (self.il6_pg_ml - 5) * 5))
        
        # IL-10 (>10 is optimal)
        if self.il10_pg_ml > 10:
            scores.append(100)
        else:
            scores.append((self.il10_pg_ml / 10) * 100)
        
        # CRP (<3 is optimal)
        if self.crp_mg_l < 3:
            scores.append(100)
        else:
            scores.append(max(0, 100 - (self.crp_mg_l - 3) * 10))
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class MetabolicBalanceAxis(HomeostasisAxis):
    """Metabolic Balance (Energy Homeostasis)."""
    
    def __init__(self):
        super().__init__(name="Metabolic Balance")
        self.blood_glucose_mg_dl = 90
        self.insulin_sensitivity_percent = 100
        self.lactate_mmol_l = 1.5
        self.atp_production_percent = 100
    
    def calculate_score(self) -> float:
        """Calculate metabolic balance score."""
        scores = []
        
        # Blood glucose (70-100 is optimal)
        if 70 <= self.blood_glucose_mg_dl <= 100:
            scores.append(100)
        elif 100 < self.blood_glucose_mg_dl <= 126:
            scores.append(70)
        else:
            scores.append(30)
        
        # Insulin sensitivity
        scores.append(self.insulin_sensitivity_percent)
        
        # ATP production
        scores.append(self.atp_production_percent)
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class ImmuneBalanceAxis(HomeostasisAxis):
    """Immune Balance (Immune Homeostasis)."""
    
    def __init__(self):
        super().__init__(name="Immune Balance")
        self.cd4_cd8_ratio = 2.0
        self.nk_cell_activity_percent = 25
        self.regulatory_t_cells_percent = 7
    
    def calculate_score(self) -> float:
        """Calculate immune balance score."""
        scores = []
        
        # CD4/CD8 ratio (1.5-2.5 is optimal)
        if 1.5 <= self.cd4_cd8_ratio <= 2.5:
            scores.append(100)
        else:
            deviation = abs(self.cd4_cd8_ratio - 2.0) / 2.0 * 100
            scores.append(max(0, 100 - deviation))
        
        # NK cell activity (>20% is optimal)
        if self.nk_cell_activity_percent > 20:
            scores.append(100)
        else:
            scores.append((self.nk_cell_activity_percent / 20) * 100)
        
        # Regulatory T cells (5-10% is optimal)
        if 5 <= self.regulatory_t_cells_percent <= 10:
            scores.append(100)
        else:
            scores.append(70)
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class ElectrolyteBalanceAxis(HomeostasisAxis):
    """Electrolyte Balance (Mineral Homeostasis)."""
    
    def __init__(self):
        super().__init__(name="Electrolyte Balance")
        self.sodium_meq_l = 140
        self.potassium_meq_l = 4.0
        self.calcium_mg_dl = 9.5
        self.magnesium_mg_dl = 2.0
    
    def calculate_score(self) -> float:
        """Calculate electrolyte balance score."""
        scores = []
        
        # Sodium (135-145)
        if 135 <= self.sodium_meq_l <= 145:
            scores.append(100)
        else:
            scores.append(50)
        
        # Potassium (3.5-5.0)
        if 3.5 <= self.potassium_meq_l <= 5.0:
            scores.append(100)
        else:
            scores.append(30)  # Critical if out of range
        
        # Magnesium (1.7-2.2)
        if 1.7 <= self.magnesium_mg_dl <= 2.2:
            scores.append(100)
        else:
            scores.append(70)
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class HormonalBalanceAxis(HomeostasisAxis):
    """Hormonal Balance (Endocrine Homeostasis)."""
    
    def __init__(self):
        super().__init__(name="Hormonal Balance")
        self.cortisol_circadian = "normal"
        self.thyroid_tsh = 2.0
        self.testosterone_ng_dl = 500  # Male reference
    
    def calculate_score(self) -> float:
        """Calculate hormonal balance score."""
        scores = []
        
        # TSH (0.5-4.5 is optimal)
        if 0.5 <= self.thyroid_tsh <= 4.5:
            scores.append(100)
        else:
            scores.append(60)
        
        # Cortisol rhythm
        if self.cortisol_circadian == "normal":
            scores.append(100)
        else:
            scores.append(50)
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class AutonomicBalanceAxis(HomeostasisAxis):
    """Autonomic Balance (Sympathetic/Parasympathetic)."""
    
    def __init__(self):
        super().__init__(name="Autonomic Balance")
        self.heart_rate_variability = 60
        self.sympathetic_tone = "balanced"
        self.parasympathetic_tone = "active"
    
    def calculate_score(self) -> float:
        """Calculate autonomic balance score."""
        scores = []
        
        # HRV (>50 is optimal)
        if self.heart_rate_variability > 50:
            scores.append(100)
        else:
            scores.append((self.heart_rate_variability / 50) * 100)
        
        # Balance
        if self.sympathetic_tone == "balanced" and self.parasympathetic_tone == "active":
            scores.append(100)
        else:
            scores.append(60)
        
        self.score = sum(scores) / len(scores)
        self.state = self.classify_state()
        return self.score


class HomeostasisEngine:
    """Measure and track homeostatic balance across all axes."""
    
    def __init__(self):
        self.axes = {
            "ph_balance": PHBalanceAxis(),
            "redox_balance": RedoxBalanceAxis(),
            "inflammatory_balance": InflammatoryBalanceAxis(),
            "metabolic_balance": MetabolicBalanceAxis(),
            "immune_balance": ImmuneBalanceAxis(),
            "electrolyte_balance": ElectrolyteBalanceAxis(),
            "hormonal_balance": HormonalBalanceAxis(),
            "autonomic_balance": AutonomicBalanceAxis()
        }
    
    def measure_all_axes(self) -> Dict[str, float]:
        """Calculate scores for all axes."""
        scores = {}
        for axis_name, axis in self.axes.items():
            scores[axis_name] = axis.calculate_score()
        return scores
    
    def calculate_overall_homeostasis(self) -> Dict[str, any]:
        """Calculate overall homeostasis score."""
        scores = self.measure_all_axes()
        overall_score = sum(scores.values()) / len(scores)
        
        imbalanced_axes = [
            name for name, score in scores.items() if score < 70
        ]
        
        return {
            "overall_score": overall_score,
            "axis_scores": scores,
            "imbalanced_axes": imbalanced_axes,
            "status": self._classify_status(overall_score)
        }
    
    def _classify_status(self, score: float) -> str:
        """Classify homeostasis status."""
        if score >= 90:
            return "EXCELLENT"
        elif score >= 70:
            return "GOOD"
        elif score >= 50:
            return "COMPROMISED"
        else:
            return "CRITICAL"
    
    def identify_restoration_targets(self) -> List[Tuple[str, float]]:
        """Identify which axes need restoration."""
        scores = self.measure_all_axes()
        targets = [
            (axis_name, score) 
            for axis_name, score in scores.items() 
            if score < 85
        ]
        return sorted(targets, key=lambda x: x[1])  # Worst first


if __name__ == "__main__":
    print("=== Homeostasis Engine Demo ===\n")
    
    engine = HomeostasisEngine()
    
    # Baseline
    print("1. BASELINE HOMEOSTASIS")
    result = engine.calculate_overall_homeostasis()
    print(f"Overall score: {result['overall_score']:.1f}/100")
    print(f"Status: {result['status']}")
    print(f"Axis scores:")
    for axis, score in result['axis_scores'].items():
        print(f"  {axis}: {score:.1f}/100")
    print()
    
    # Simulate chronic inflammation
    print("2. CHRONIC INFLAMMATION STATE")
    engine.axes["inflammatory_balance"].il6_pg_ml = 25
    engine.axes["inflammatory_balance"].crp_mg_l = 15
    engine.axes["redox_balance"].glutathione_ratio = 5
    engine.axes["redox_balance"].mitochondrial_function_percent = 60
    
    result = engine.calculate_overall_homeostasis()
    print(f"Overall score: {result['overall_score']:.1f}/100")
    print(f"Status: {result['status']}")
    print(f"Imbalanced axes: {result['imbalanced_axes']}")
    print()
    
    # Identify targets
    print("3. RESTORATION TARGETS")
    targets = engine.identify_restoration_targets()
    print("Axes needing restoration (worst first):")
    for axis_name, score in targets:
        print(f"  {axis_name}: {score:.1f}/100")
