#!/usr/bin/env python3
"""
Medical NanoCERN - Flagship Drug KUs
Phase 1: NSAIDs + Critical Drugs (Full Set of 10)

Manually curated medical KUs with safety envelopes for:
1. Paracetamol (Hepatotoxicity)
2. Ibuprofen (Renal/GI)
3. Warfarin (Therapeutic Window)
4. Aspirin (Bleeding)
5. Metformin (Lactic Acidosis)
6. Diclofenac (CV/Hepatic)
7. Naproxen (GI/Renal)
8. Amoxicillin (Allergy/Renal)
9. Digoxin (Narrow Index)
10. Morphine (Respiratory/CNS)
"""

from dataclasses import dataclass, field
from typing import Dict, List, Optional, Any
import json

@dataclass(frozen=True)
class MedicalKU:
    """Medical Knowledge Unit with safety envelopes."""
    id: str
    domain: str
    type: str  # pharmacokinetics, pharmacodynamics, safety
    drug_name: str
    
    invariant: str
    applies_if: Dict[str, Any]
    failure_modes: List[str]
    
    # Medical-specific fields
    contraindications_absolute: List[str] = field(default_factory=list)
    contraindications_relative: List[str] = field(default_factory=list)
    interactions: List[Dict[str, str]] = field(default_factory=list)
    
    metadata: Dict[str, Any] = field(default_factory=dict)


# Flagship Medical KUs (10 Total)
FLAGSHIP_MEDICAL_KUS = [
    # 1. Paracetamol (Acetaminophen) - Hepatotoxicity
    MedicalKU(
        id="MED-PARACETAMOL-HEPATOTOX",
        domain="medicine",
        type="pharmacokinetics",
        drug_name="Paracetamol",
        invariant="Paracetamol clearance depends on liver function and dose",
        applies_if={
            "dose_mg_per_kg <": 75,
            "daily_dose_mg <": 4000,
            "liver_function": "normal",
            "alcohol_use": False
        },
        failure_modes=[
            "hepatotoxicity",
            "acute_liver_failure",
            "glutathione_depletion"
        ],
        contraindications_absolute=[
            "severe_liver_disease",
            "active_alcoholism"
        ],
        contraindications_relative=[
            "chronic_alcohol_use",
            "hepatitis",
            "gilbert_syndrome"
        ],
        interactions=[
            {
                "drug": "warfarin",
                "severity": "moderate",
                "effect": "increased_INR"
            },
            {
                "drug": "alcohol",
                "severity": "major",
                "effect": "hepatotoxicity_risk"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "narrow",
            "evidence_level": "1A",
            "ld50_human_mg_kg": 150
        }
    ),
    
    # 2. Ibuprofen - Renal + GI toxicity
    MedicalKU(
        id="MED-IBUPROFEN-RENAL",
        domain="medicine",
        type="pharmacokinetics",
        drug_name="Ibuprofen",
        invariant="Ibuprofen clearance requires normal renal function",
        applies_if={
            "dose_mg <": 800,
            "daily_dose_mg <": 3200,
            "egfr_ml_min >": 60,
            "age <": 65,
            "no_gi_ulcer": True
        },
        failure_modes=[
            "acute_kidney_injury",
            "gi_bleeding",
            "cardiovascular_events"
        ],
        contraindications_absolute=[
            "severe_renal_impairment",
            "active_gi_bleeding",
            "aspirin_allergy"
        ],
        contraindications_relative=[
            "hypertension",
            "heart_failure",
            "elderly_>75"
        ],
        interactions=[
            {
                "drug": "aspirin",
                "severity": "major",
                "effect": "reduced_cardioprotection"
            },
            {
                "drug": "warfarin",
                "severity": "major",
                "effect": "bleeding_risk"
            },
            {
                "drug": "ACE_inhibitors",
                "severity": "moderate",
                "effect": "reduced_antihypertensive_effect"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "moderate",
            "evidence_level": "1A",
            "max_otc_dose_mg": 1200
        }
    ),
    
    # 3. Warfarin - Narrow therapeutic window
    MedicalKU(
        id="MED-WARFARIN-INR",
        domain="medicine",
        type="pharmacodynamics",
        drug_name="Warfarin",
        invariant="Warfarin anticoagulation requires INR monitoring",
        applies_if={
            "inr >=": 2.0,
            "inr <=": 3.0,
            "liver_function": "normal",
            "no_bleeding_risk": True,
            "vitamin_k_intake_stable": True
        },
        failure_modes=[
            "major_bleeding",
            "intracranial_hemorrhage",
            "thrombosis_if_subtherapeutic"
        ],
        contraindications_absolute=[
            "active_bleeding",
            "severe_liver_disease",
            "pregnancy",
            "recent_cns_surgery"
        ],
        contraindications_relative=[
            "falls_risk",
            "elderly_>80",
            "uncontrolled_hypertension"
        ],
        interactions=[
            {
                "drug": "aspirin",
                "severity": "major",
                "effect": "bleeding_risk"
            },
            {
                "drug": "NSAIDs",
                "severity": "major",
                "effect": "gi_bleeding"
            },
            {
                "drug": "antibiotics",
                "severity": "major",
                "effect": "INR_elevation"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "very_narrow",
            "evidence_level": "1A",
            "monitoring_required": "weekly_INR"
        }
    ),
    
    # 4. Aspirin - Bleeding risk
    MedicalKU(
        id="MED-ASPIRIN-BLEEDING",
        domain="medicine",
        type="pharmacodynamics",
        drug_name="Aspirin",
        invariant="Aspirin irreversibly inhibits platelet function",
        applies_if={
            "dose_mg <=": 325,
            "age >=": 18,
            "no_bleeding_disorder": True,
            "no_active_ulcer": True
        },
        failure_modes=[
            "gi_bleeding",
            "intracranial_hemorrhage",
            "reye_syndrome_pediatric"
        ],
        contraindications_absolute=[
            "hemophilia",
            "active_peptic_ulcer",
            "children_<12_with_fever"
        ],
        contraindications_relative=[
            "asthma",
            "gout",
            "elderly_>75"
        ],
        interactions=[
            {
                "drug": "warfarin",
                "severity": "major",
                "effect": "bleeding_risk"
            },
            {
                "drug": "ibuprofen",
                "severity": "major",
                "effect": "reduced_cardioprotection"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "moderate",
            "evidence_level": "1A",
            "irreversible_effect": True
        }
    ),
    
    # 5. Metformin - Lactic acidosis
    MedicalKU(
        id="MED-METFORMIN-LACTATE",
        domain="medicine",
        type="pharmacokinetics",
        drug_name="Metformin",
        invariant="Metformin clearance requires normal renal function",
        applies_if={
            "egfr_ml_min >=": 30,
            "no_metabolic_acidosis": True,
            "no_heart_failure": True,
            "daily_dose_mg <=": 2000
        },
        failure_modes=[
            "lactic_acidosis",
            "acute_kidney_injury"
        ],
        contraindications_absolute=[
            "severe_renal_impairment",
            "metabolic_acidosis",
            "diabetic_ketoacidosis"
        ],
        contraindications_relative=[
            "heart_failure",
            "liver_disease",
            "alcohol_abuse"
        ],
        interactions=[
            {
                "drug": "iodinated_contrast",
                "severity": "major",
                "effect": "lactic_acidosis_risk"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "wide",
            "evidence_level": "1A",
            "hold_before_contrast": "48_hours"
        }
    ),
    
    # 6. Diclofenac - Cardiovascular & Hepatic Risk
    MedicalKU(
        id="MED-DICLOFENAC-CV-LIVER",
        domain="medicine",
        type="safety",
        drug_name="Diclofenac",
        invariant="COX-2 inhibition increases cardiovascular thrombotic risk",
        applies_if={
            "daily_dose_mg <": 150,
            "liver_function": "normal",
            "no_heart_failure": True,
            "no_history_mi": True,
            "no_active_gi_bleed": True
        },
        failure_modes=[
            "hepatotoxicity",
            "myocardial_infarction",
            "stroke",
            "gi_perforation"
        ],
        contraindications_absolute=[
            "coronary_artery_bypass_graft_surgery",
            "severe_liver_failure",
            "active_gi_bleeding"
        ],
        contraindications_relative=[
            "hypertension",
            "dyslipidemia",
            "smoker"
        ],
        interactions=[
             {
                "drug": "aspirin",
                "severity": "major",
                "effect": "bleeding_risk_and_reduced_cardioprotection"
            },
            {
                "drug": "warfarin",
                "severity": "major",
                "effect": "bleeding_risk"
            }
        ],
        metadata={
            "fda_approved": True,
            "box_warning": "cardiovascular_thrombotic_events",
            "evidence_level": "1A"
        }
    ),
    
    # 7. Naproxen - GI & Renal (Safer CV profile)
    MedicalKU(
        id="MED-NAPROXEN-GI",
        domain="medicine",
        type="pharmacokinetics",
        drug_name="Naproxen",
        invariant="Non-selective COX inhibition affects gastric mucosa and renal blood flow",
        applies_if={
            "daily_dose_mg <=": 1000,
            "egfr_ml_min >": 30,
            "no_gi_ulcer": True
        },
        failure_modes=[
            "gi_bleeding",
            "ulceration",
            "interstitial_nephritis"
        ],
        contraindications_absolute=[
            "active_peptic_ulcer",
            "aspirin_allergy",
            "recent_cabg"
        ],
        contraindications_relative=[
            "elderly_>65",
            "concurrent_corticosteroids"
        ],
        interactions=[
            {
                "drug": "lithium",
                "severity": "moderate",
                "effect": "increased_lithium_levels"
            },
            {
                "drug": "methotrexate",
                "severity": "major",
                "effect": "methotrexate_toxicity"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "moderate",
            "cv_risk": "lowest_of_nsaids"
        }
    ),
    
    # 8. Amoxicillin - Allergy & Renal
    MedicalKU(
        id="MED-AMOXICILLIN-ALLERGY",
        domain="medicine",
        type="pharmacokinetics",
        drug_name="Amoxicillin",
        invariant="Beta-lactam ring stability + renal elimination",
        applies_if={
            "no_penicillin_allergy": True,
            "egfr_ml_min >": 30  # Needs adjustment below this, but for simplicity strict > 30 for std dose
        },
        failure_modes=[
            "anaphylaxis",
            "hypersensitivity_reaction",
            "seizures_accumulation"
        ],
        contraindications_absolute=[
            "history_of_penicillin_anaphylaxis",
            "stevens_johnson_syndrome_history"
        ],
        contraindications_relative=[
            "cephalosporin_allergy_mild",
            "mononucleosis" # rash risk
        ],
        interactions=[
            {
                "drug": "allopurinol",
                "severity": "moderate",
                "effect": "increased_rash_risk"
            },
            {
                "drug": "warfarin",
                "severity": "minor",
                "effect": "possible_increased_INR"
            }
        ],
        metadata={
            "fda_approved": True,
            "pregnancy_category": "B",
            "safe_in_pregnancy": True
        }
    ),
    
    # 9. Digoxin - Narrow Therapeutic Index
    MedicalKU(
        id="MED-DIGOXIN-TOXICITY",
        domain="medicine",
        type="pharmacokinetics",
        drug_name="Digoxin",
        invariant="Na+/K+-ATPase inhibition requires precise serum concentration",
        applies_if={
            "daily_dose_mg <=": 0.25,
            "egfr_ml_min >": 50,
            "serum_potassium >": 3.5,
            "no_av_block": True
        },
        failure_modes=[
            "digoxin_toxicity",
            "ventricular_arrhythmias",
            "heart_block"
        ],
        contraindications_absolute=[
            "ventricular_fibrillation",
            "hypersensitivity"
        ],
        contraindications_relative=[
            "hypokalemia",
            "hypomagnesemia",
            "hypercalcemia",
            "wolff_parkinson_white"
        ],
        interactions=[
            {
                "drug": "amiodarone",
                "severity": "major",
                "effect": "increased_digoxin_levels"
            },
            {
                "drug": "verapamil",
                "severity": "major",
                "effect": "increased_digoxin_levels"
            },
            {
                "drug": "loop_diuretics",
                "severity": "moderate",
                "effect": "hypokalemia_risk"
            }
        ],
        metadata={
            "fda_approved": True,
            "therapeutic_index": "narrow",
            "monitoring_required": "serum_levels_and_electrolytes"
        }
    ),
    
    # 10. Morphine - Respiratory Depression
    MedicalKU(
        id="MED-MORPHINE-RESP",
        domain="medicine",
        type="pharmacodynamics",
        drug_name="Morphine",
        invariant="Mu-opioid receptor activation depresses respiratory drive",
        applies_if={
            "respiratory_rate >": 12,
            "no_cns_depressants": True,
            "egfr_ml_min >": 30, # Accumulation of M6G
            "no_paralytic_ileus": True
        },
        failure_modes=[
            "respiratory_arrest",
            "coma",
            "hypotension"
        ],
        contraindications_absolute=[
            "severe_respiratory_depression",
            "acute_severe_asthma",
            "paralytic_ileus"
        ],
        contraindications_relative=[
            "head_injury",
            "raised_intracranial_pressure",
            "biliary_tract_disease"
        ],
        interactions=[
            {
                "drug": "benzodiazepines",
                "severity": "critical",
                "effect": "profound_sedation_respiratory_depression"
            },
            {
                "drug": "alcohol",
                "severity": "critical",
                "effect": "fatal_overdose_risk"
            },
            {
                "drug": "MAO_inhibitors",
                "severity": "major",
                "effect": "serotonin_syndrome_or_toxicity"
            }
        ],
        metadata={
            "fda_approved": True,
            "schedule": "II",
            "antidote": "naloxone"
        }
    )
]


def save_medical_kus():
    """Save medical KUs to JSON."""
    
    medical_data = []
    for ku in FLAGSHIP_MEDICAL_KUS:
        medical_data.append({
            'id': ku.id,
            'domain': ku.domain,
            'type': ku.type,
            'drug_name': ku.drug_name,
            'invariant': ku.invariant,
            'applies_if': ku.applies_if,
            'failure_modes': ku.failure_modes,
            'contraindications_absolute': ku.contraindications_absolute,
            'contraindications_relative': ku.contraindications_relative,
            'interactions': ku.interactions,
            'metadata': ku.metadata
        })
    
    with open('flagship_medical_kus.json', 'w') as f:
        json.dump(medical_data, f, indent=2)
    
    print(f"✓ Saved {len(FLAGSHIP_MEDICAL_KUS)} medical KUs to flagship_medical_kus.json")


if __name__ == "__main__":
    save_medical_kus()
    
    print("\nFlagship Medical KUs:")
    for ku in FLAGSHIP_MEDICAL_KUS:
        print(f"\n  {ku.id}: {ku.drug_name}")
        print(f"    Type: {ku.type}")
        print(f"    Envelopes: {len(ku.applies_if)} constraints")
        print(f"    Absolute contraindications: {len(ku.contraindications_absolute)}")
