#!/usr/bin/env python3
"""
Homeostasis-Focused Drug Discovery Engine
Discovers drugs that restore physiological balance.
"""

from body_orchestrator import BodySystemOrchestrator
from homeostasis_engine import HomeostasisEngine
from typing import Dict, List, Tuple, Any
import json


class DrugCandidate:
    """Represents a drug candidate with homeostatic effects."""
    
    def __init__(self, name: str, drug_class: str):
        self.name = name
        self.drug_class = drug_class  # A, B, C, or D
        self.homeostasis_score = 0
        
        # Homeostatic impact
        self.axes_restored = []
        self.axes_supported = []
        self.axes_disrupted = []
        
        # Healing properties
        self.healing_acceleration = 1.0
        self.mechanisms = []
        self.targets = []
        
        # Safety
        self.organ_toxicity = []
        self.therapeutic_window = 1.0
    
    def calculate_homeostasis_score(self) -> int:
        """Calculate homeostasis score based on impact."""
        score = (
            len(self.axes_restored) * 3 +
            len(self.axes_supported) * 1 +
            len(self.axes_disrupted) * (-2)
        )
        self.homeostasis_score = score
        return score
    
    def classify(self) -> str:
        """Classify drug based on homeostasis score."""
        if self.homeostasis_score >= 6:
            self.drug_class = "A"  # Homeostasis Restorer
        elif self.homeostasis_score >= 0:
            self.drug_class = "B"  # Homeostasis Neutral
        elif self.homeostasis_score >= -5:
            self.drug_class = "C"  # Homeostasis Disruptor
        else:
            self.drug_class = "D"  # Homeostasis Destroyer
        
        return self.drug_class


# Healing Drug Database
HEALING_DRUGS = [
    {
        "name": "Magnesium Glycinate",
        "class": "A",
        "axes_restored": ["electrolyte_balance", "metabolic_balance", "autonomic_balance"],
        "axes_supported": ["inflammatory_balance", "redox_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.3,
        "mechanisms": ["ATP synthesis cofactor", "NMDA modulation", "Parasympathetic activation"],
        "targets": ["mitochondria", "nervous_system", "muscles"],
        "organ_toxicity": [],
        "therapeutic_window": 10.0
    },
    {
        "name": "Omega-3 EPA/DHA",
        "class": "A",
        "axes_restored": ["inflammatory_balance", "redox_balance", "metabolic_balance"],
        "axes_supported": ["immune_balance", "hormonal_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.5,
        "mechanisms": ["SPM production", "Membrane fluidity", "Anti-inflammatory signaling"],
        "targets": ["cell_membranes", "immune_system", "brain"],
        "organ_toxicity": [],
        "therapeutic_window": 20.0
    },
    {
        "name": "Vitamin D3",
        "class": "A",
        "axes_restored": ["immune_balance", "hormonal_balance"],
        "axes_supported": ["inflammatory_balance", "metabolic_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.4,
        "mechanisms": ["Immune modulation", "Calcium homeostasis", "Gene expression"],
        "targets": ["immune_system", "bones", "muscles"],
        "organ_toxicity": [],
        "therapeutic_window": 5.0
    },
    {
        "name": "CoQ10 (Ubiquinone)",
        "class": "A",
        "axes_restored": ["metabolic_balance", "redox_balance"],
        "axes_supported": ["autonomic_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.3,
        "mechanisms": ["Mitochondrial ETC", "Antioxidant", "ATP production"],
        "targets": ["mitochondria", "heart", "brain"],
        "organ_toxicity": [],
        "therapeutic_window": 15.0
    },
    {
        "name": "Curcumin",
        "class": "A",
        "axes_restored": ["inflammatory_balance", "redox_balance"],
        "axes_supported": ["metabolic_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.4,
        "mechanisms": ["NF-κB inhibition", "Nrf2 activation", "Antioxidant"],
        "targets": ["immune_system", "liver", "joints"],
        "organ_toxicity": [],
        "therapeutic_window": 10.0
    },
    {
        "name": "Berberine",
        "class": "A",
        "axes_restored": ["metabolic_balance", "inflammatory_balance"],
        "axes_supported": ["redox_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.3,
        "mechanisms": ["AMPK activation", "Glucose metabolism", "Lipid regulation"],
        "targets": ["liver", "pancreas", "gut"],
        "organ_toxicity": [],
        "therapeutic_window": 8.0
    },
    {
        "name": "Alpha-Lipoic Acid",
        "class": "A",
        "axes_restored": ["redox_balance", "metabolic_balance"],
        "axes_supported": ["inflammatory_balance"],
        "axes_disrupted": [],
        "healing_acceleration": 1.2,
        "mechanisms": ["Universal antioxidant", "Mitochondrial support", "Insulin sensitizer"],
        "targets": ["mitochondria", "nerves", "liver"],
        "organ_toxicity": [],
        "therapeutic_window": 12.0
    }
]


class HomeostasisDrugDiscovery:
    """Discover drugs that restore homeostatic balance."""
    
    def __init__(self):
        self.body = BodySystemOrchestrator()
        self.homeostasis = HomeostasisEngine()
        self.healing_drugs = HEALING_DRUGS
    
    def discover_for_condition(self, condition: str, patient_profile: Dict = None) -> List[Dict]:
        """
        Find optimal drugs/combinations for a specific condition.
        
        Strategy:
        1. Simulate disease state
        2. Identify which homeostatic axes are imbalanced
        3. Find drugs that restore those specific axes
        4. Rank by healing acceleration and safety
        """
        
        # Step 1: Simulate disease
        disease_result = self.body.simulate_disease_progression(condition)
        
        # Step 2: Identify imbalances (organ-level)
        homeostasis_result = disease_result["whole_body_homeostasis"]
        imbalanced_organs = homeostasis_result.get("imbalanced_organs", [])
        
        # Map organs to axes they affect
        organ_to_axes = {
            "heart": ["metabolic_balance", "autonomic_balance"],
            "kidneys": ["electrolyte_balance", "ph_balance"],
            "brain": ["autonomic_balance", "hormonal_balance"],
            "liver": ["metabolic_balance", "redox_balance"],
            "pancreas": ["metabolic_balance", "hormonal_balance"]
        }
        
        imbalanced_axes = []
        for organ in imbalanced_organs:
            imbalanced_axes.extend(organ_to_axes.get(organ, []))
        imbalanced_axes = list(set(imbalanced_axes))  # Remove duplicates
        
        print(f"\n=== Drug Discovery for {condition.upper()} ===")
        print(f"Imbalanced organs: {imbalanced_organs}")
        print(f"Affected axes: {imbalanced_axes}")
        print(f"Overall homeostasis: {homeostasis_result['overall_homeostasis']:.1f}/100\n")
        
        # Step 3: Find restoring drugs
        candidates = []
        for drug in self.healing_drugs:
            # Check if drug restores any imbalanced axes
            restored_axes = set(drug["axes_restored"])
            target_axes = set(imbalanced_axes)
            
            if restored_axes & target_axes:  # Intersection
                match_score = len(restored_axes & target_axes)
                candidates.append({
                    "drug": drug,
                    "axes_restored": list(restored_axes & target_axes),
                    "match_score": match_score,
                    "homeostasis_score": len(drug["axes_restored"]) * 3 + len(drug["axes_supported"])
                })
        
        # Step 4: Rank by effectiveness
        ranked = sorted(candidates, 
                       key=lambda x: (x["match_score"], x["homeostasis_score"]), 
                       reverse=True)
        
        return {
            "condition": condition,
            "disease_state": disease_result,
            "imbalanced_axes": imbalanced_axes,
            "recommended_drugs": ranked[:5],  # Top 5
            "baseline_homeostasis": homeostasis_result["overall_homeostasis"]
        }
    
    def test_drug_combination(self, drugs: List[str]) -> Dict:
        """Test a combination of drugs for synergy and safety."""
        
        # Get baseline
        baseline = self.body.calculate_whole_body_homeostasis()
        
        # Apply each drug
        for drug_name in drugs:
            self.body.apply_drug_intervention(drug_name)
        
        # Get result
        after = self.body.calculate_whole_body_homeostasis()
        
        improvement = after["overall_homeostasis"] - baseline["overall_homeostasis"]
        
        return {
            "combination": drugs,
            "baseline_homeostasis": baseline["overall_homeostasis"],
            "after_homeostasis": after["overall_homeostasis"],
            "improvement": improvement,
            "synergy": "YES" if improvement > len(drugs) * 1.5 else "NO"
        }
    
    def generate_healing_protocol(self, condition: str) -> Dict:
        """Generate complete healing protocol for a condition."""
        
        discovery = self.discover_for_condition(condition)
        
        # Build protocol
        protocol = {
            "condition": condition,
            "primary_drugs": [],
            "supportive_drugs": [],
            "lifestyle": [],
            "timeline": "",
            "expected_improvement": 0
        }
        
        # Select primary drugs (top 2-3 that restore different axes)
        selected_axes = set()
        for candidate in discovery["recommended_drugs"]:
            if len(protocol["primary_drugs"]) < 3:
                drug = candidate["drug"]
                axes = set(candidate["axes_restored"])
                
                # Avoid redundancy
                if not (axes & selected_axes):
                    protocol["primary_drugs"].append(drug["name"])
                    selected_axes.update(axes)
        
        # Add supportive drugs
        for candidate in discovery["recommended_drugs"][3:5]:
            protocol["supportive_drugs"].append(candidate["drug"]["name"])
        
        # Estimate improvement
        protocol["expected_improvement"] = len(protocol["primary_drugs"]) * 5
        
        return protocol


if __name__ == "__main__":
    print("=== Homeostasis-Focused Drug Discovery Demo ===\n")
    
    discovery = HomeostasisDrugDiscovery()
    
    # Test 1: Heart Failure
    print("\n" + "="*60)
    print("TEST 1: HEART FAILURE")
    print("="*60)
    result = discovery.discover_for_condition("heart_failure")
    
    print("Recommended Drugs:")
    for i, candidate in enumerate(result["recommended_drugs"], 1):
        drug = candidate["drug"]
        print(f"\n{i}. {drug['name']} (Class {drug['class']})")
        print(f"   Restores: {', '.join(candidate['axes_restored'])}")
        print(f"   Mechanisms: {', '.join(drug['mechanisms'][:2])}")
        print(f"   Healing acceleration: {drug['healing_acceleration']}×")
        print(f"   Safety: Therapeutic window {drug['therapeutic_window']}×")
    
    # Test 2: Alzheimer's
    print("\n" + "="*60)
    print("TEST 2: ALZHEIMER'S DISEASE")
    print("="*60)
    result = discovery.discover_for_condition("alzheimers")
    
    print("Recommended Drugs:")
    for i, candidate in enumerate(result["recommended_drugs"], 1):
        drug = candidate["drug"]
        print(f"\n{i}. {drug['name']} (Class {drug['class']})")
        print(f"   Restores: {', '.join(candidate['axes_restored'])}")
        print(f"   Mechanisms: {', '.join(drug['mechanisms'][:2])}")
    
    # Test 3: Generate Healing Protocol
    print("\n" + "="*60)
    print("TEST 3: COMPLETE HEALING PROTOCOL")
    print("="*60)
    protocol = discovery.generate_healing_protocol("heart_failure")
    
    print(f"\nHealing Protocol for {protocol['condition'].upper()}:")
    print(f"\nPrimary Drugs:")
    for drug in protocol["primary_drugs"]:
        print(f"  • {drug}")
    
    print(f"\nSupportive Drugs:")
    for drug in protocol["supportive_drugs"]:
        print(f"  • {drug}")
    
    print(f"\nExpected Improvement: +{protocol['expected_improvement']} points in homeostasis score")
    
    # Save results
    with open("drug_discovery_results.json", "w") as f:
        json.dump(protocol, f, indent=2)
    print("\n✓ Results saved to drug_discovery_results.json")
