#!/usr/bin/env python3
"""
Knowledge Reactor Core (v1.0)
Deterministic Knowledge Physics Substrate
"""

import time
from dataclasses import dataclass, field
from typing import Dict, List, Optional, Tuple, Any
import json
from kuat_binary import parse_kuat # Import Phase II dependency

@dataclass(frozen=True)
class KU:
    """Atomic Knowledge Unit (Stable Invariant)"""
    id: str
    domain: str
    invariant: str  # e.g., "V = I * R"
    applies_if: Dict[str, Any]  # Envelope: {"v <": 5.0, "T <": 300}
    failure_modes: List[str] = field(default_factory=list)
    metadata: Dict[str, Any] = field(default_factory=dict)

@dataclass(frozen=True)
class NKU:
    """Negative Knowledge Unit (Structural Rejection Fragment)"""
    id: str
    rejected_ku_id: str
    reason: str
    violated_envelope: str
    state_snapshot: Dict[str, float]
    timestamp: float = field(default_factory=time.time)

@dataclass
class State:
    """System State representing a point in the knowledge manifold"""
    variables: Dict[str, float]
    timestamp: float = field(default_factory=time.time)

class MinimalReactor:
    """
    Experimental knowledge collider.
    Enforces invariance and applicability envelopes.
    """
    
    def __init__(self):
        self.registry: Dict[str, KU] = {}
        self.nku_log: List[NKU] = []
        self.cycle_count = 0
        
    def load_ku(self, ku: KU):
        """Inject a high-purity atom into the core."""
        self.registry[ku.id] = ku

    def load_kuat_atom(self, binary_data: bytes) -> str:
        """
        Inject a binary KUAT atom into the core.
        Returns the KU ID.
        """
        header_info, ku_dict = parse_kuat(binary_data)
        # Convert dict to KU object
        ku = KU(
            id=ku_dict['id'],
            domain=ku_dict['domain'],
            invariant=ku_dict['invariant'],
            applies_if=ku_dict['applies_if'],
            failure_modes=ku_dict.get('failure_modes', []),
            metadata=ku_dict.get('metadata', {})
        )
        self.load_ku(ku)
        return ku.id
        
    def validate_reaction(self, ku_id: str, state: State) -> Tuple[bool, Optional[str]]:
        """
        Collision Verification:
        Checks if the state is within the KU's applicability envelope.
        """
        ku = self.registry.get(ku_id)
        if not ku:
            return False, f"KU_NOT_FOUND: {ku_id}"
            
        for condition, threshold in ku.applies_if.items():
            # Parse simple condition "var op"
            parts = condition.split()
            
            # Handle implicit equality / boolean flags (e.g. "no_penicillin_allergy": True)
            if len(parts) == 1:
                var_name = parts[0]
                op = '==' 
            elif len(parts) >= 2:
                var_name = parts[0]
                op = parts[1]
            else:
                continue
            

            
            if var_name not in state.variables:
                return False, f"MISSING_VAR: {var_name}"
            
            val = state.variables[var_name]
            
            # Deterministic gating
            if op == '<' and not (val < threshold):
                return False, f"ENVELOPE_VIOLATION: {var_name}({val}) < {threshold} is FALSE"
            elif op == '>' and not (val > threshold):
                return False, f"ENVELOPE_VIOLATION: {var_name}({val}) > {threshold} is FALSE"
            elif op == '<=' and not (val <= threshold):
                return False, f"ENVELOPE_VIOLATION: {var_name}({val}) <= {threshold} is FALSE"
            elif op == '>=' and not (val >= threshold):
                return False, f"ENVELOPE_VIOLATION: {var_name}({val}) >= {threshold} is FALSE"
            elif op == '==' and not (val == threshold):
                return False, f"ENVELOPE_VIOLATION: {var_name}({val}) == {threshold} is FALSE"
            elif op == '!=' and not (val != threshold):
                return False, f"ENVELOPE_VIOLATION: {var_name}({val}) != {threshold} is FALSE"
                
        return True, None

    def execute_reaction(self, ku_id: str, state: State) -> Tuple[Optional[State], Optional[NKU]]:
        """
        Attempt a knowledge collision.
        Returns (NewState, None) on Success.
        Returns (None, NKU) on Structural Rejection.
        """
        self.cycle_count += 1
        ku = self.registry.get(ku_id)
        
        # GATE 1: Envelope Validation
        is_valid, reason = self.validate_reaction(ku_id, state)
        
        if not is_valid:
            # COLLISION FAILURE: Emit NKU
            nku_id = f"NKU-{int(time.time())}-{self.cycle_count}"
            nku = NKU(
                id=nku_id,
                rejected_ku_id=ku_id,
                reason="Structural Rejection: Application outside envelope",
                violated_envelope=reason or "Unknown",
                state_snapshot=state.variables.copy()
            )
            self.nku_log.append(nku)
            return None, nku
            
        # GATE 2: Transition (Physics Layer)
        # For the minimal reactor, we represent this as a state propagation
        # In full version, this involves R-BC execution of the invariant.
        new_vars = state.variables.copy()
        
        # Simulate invariant side-effect (e.g. if we are applying Ohm's law to find V)
        # Note: In Spec 1.0, transition logic is internal to invariant application.
        return State(variables=new_vars), None

    def get_audit_trail(self) -> List[Dict]:
        """Return history of NKU emissions."""
        return [vars(n) for n in self.nku_log]

if __name__ == "__main__":
    print("✓ Knowledge Reactor Core initialized (Minimal Mode)")
