# Knowledge Unit Library

## Overview

This package includes **231 Knowledge Units (KUs)** across 4 scientific domains:

- **Medical**: 90 KUs - Drug safety, contraindications, physiological constraints
- **Physics**: 50 KUs - Physical laws, conservation principles, regime boundaries
- **Chemistry**: 41 KUs - Reaction constraints, safety limits, thermodynamic bounds
- **Mathematics**: 50 KUs - Theorem applicability, numerical stability, convergence criteria

---

## Directory Structure

```
nanocern_reactor/
├── generated_kus/
│   ├── medical/
│   │   └── medical_kus.json          # 90 medical KUs
│   ├── physics/
│   │   └── physics_kus.json          # 50 physics KUs
│   ├── chemistry/
│   │   └── chemistry_kus.json        # 41 chemistry KUs
│   └── mathematics/
│       └── mathematics_kus.json      # 50 mathematics KUs
│
├── atoms/                             # Individual .ku files (if any)
└── flagship_medical_kus.py           # Curated medical safety KUs
```

---

## Knowledge Unit Format

Each KU follows this structure:

```json
{
  "id": "MED-DRUG-001",
  "domain": "medicine",
  "invariant": "Drug X contraindicated when condition Y",
  "applies_if": {
    "parameter_1 <": threshold_1,
    "parameter_2 >": threshold_2
  },
  "failure_modes": [
    "adverse_effect_1",
    "adverse_effect_2"
  ],
  "metadata": {
    "source": "FDA/literature/generated",
    "confidence": 0.95
  }
}
```

---

## Usage Examples

### Loading Medical KUs

```python
import json

# Load medical knowledge units
with open('nanocern_reactor/generated_kus/medical/medical_kus.json', 'r') as f:
    medical_kus = json.load(f)

print(f"Loaded {len(medical_kus)} medical KUs")

# Example: Check drug safety
patient_state = {
    "egfr_ml_min": 25,  # Low kidney function
    "age_years": 75,
    "dose_mg_per_kg": 50
}

# Find applicable KUs
for ku in medical_kus:
    # Check if KU applies to this patient
    applies = True
    for constraint, threshold in ku['applies_if'].items():
        param_name = constraint.split()[0]
        operator = constraint.split()[1]
        
        if param_name in patient_state:
            # Evaluate constraint
            # (simplified - use proper constraint evaluation)
            pass
```

### Loading All Domains

```python
import json
from pathlib import Path

# Load all KU domains
ku_library = {}
domains = ['medical', 'physics', 'chemistry', 'mathematics']

for domain in domains:
    ku_file = f'nanocern_reactor/generated_kus/{domain}/{domain}_kus.json'
    with open(ku_file, 'r') as f:
        ku_library[domain] = json.load(f)
    print(f"{domain.capitalize()}: {len(ku_library[domain])} KUs")

# Total KUs
total_kus = sum(len(kus) for kus in ku_library.values())
print(f"\nTotal KUs: {total_kus}")
```

---

## Medical KUs (90 total)

### Categories

1. **Drug Safety** - Contraindications, dosing limits, interactions
2. **Physiological Constraints** - Organ function thresholds
3. **Age-Related** - Pediatric and geriatric considerations
4. **Renal Function** - GFR-based dosing adjustments
5. **Hepatic Function** - Liver metabolism considerations

### Example Medical KUs

- Lisinopril dosing constraints (eGFR > 30 mL/min)
- Amlodipine safety thresholds
- Metoprolol contraindications
- Warfarin INR monitoring requirements
- Metformin renal safety limits

---

## Physics KUs (50 total)

### Categories

1. **Conservation Laws** - Energy, momentum, charge
2. **Thermodynamics** - Entropy, temperature limits
3. **Mechanics** - Newton's laws, stress-strain
4. **Electromagnetism** - Maxwell's equations, field limits
5. **Quantum Mechanics** - Uncertainty, superposition

### Example Physics KUs

- Energy conservation in closed systems
- Momentum conservation in collisions
- Ohm's law validity regime
- Ideal gas law applicability
- Hooke's law elastic limits

---

## Chemistry KUs (41 total)

### Categories

1. **Reaction Safety** - Temperature, pressure limits
2. **Thermodynamics** - Gibbs free energy, equilibrium
3. **Kinetics** - Rate laws, activation energy
4. **Stoichiometry** - Mass balance, limiting reagents
5. **Safety Constraints** - Explosive limits, toxicity

### Example Chemistry KUs

- Reaction temperature safety limits
- pH constraints for stability
- Concentration limits for safety
- Pressure vessel constraints
- Explosive mixture boundaries

---

## Mathematics KUs (50 total)

### Categories

1. **Numerical Methods** - Convergence, stability
2. **Calculus** - Differentiation, integration limits
3. **Linear Algebra** - Matrix operations, eigenvalues
4. **Statistics** - Sample size, confidence intervals
5. **Optimization** - Convexity, constraint satisfaction

### Example Mathematics KUs

- Numerical stability conditions
- Convergence criteria for iterative methods
- Matrix invertibility requirements
- Statistical significance thresholds
- Optimization constraint boundaries

---

## Using KUs with NanoCERN Reactor

### Basic Validation

```python
from nanocern_reactor.core import MinimalReactor
import json

# Load KUs
with open('nanocern_reactor/generated_kus/medical/medical_kus.json', 'r') as f:
    medical_kus = json.load(f)

# Initialize reactor
reactor = MinimalReactor()

# Patient state
patient_state = {
    "egfr_ml_min": 25,
    "age_years": 75,
    "dose_mg_per_kg": 50
}

# Execute reaction (validate state)
result = reactor.execute_reaction(
    state=patient_state,
    kus=medical_kus
)

# Check violations
if result.get('nkus'):
    print("⚠️ Contraindications detected:")
    for nku in result['nkus']:
        print(f"  - {nku['reason']}")
else:
    print("✅ All constraints satisfied")
```

---

## KU Quality Levels

### Source Types

1. **FDA/EMA Approved** - Regulatory agency guidelines (highest confidence)
2. **Clinical Literature** - Peer-reviewed medical journals
3. **Generated** - Algorithmically generated from patterns (requires validation)

### Confidence Scores

- **0.95-1.0**: Regulatory or well-established
- **0.85-0.94**: Strong clinical evidence
- **0.70-0.84**: Moderate evidence, requires validation
- **< 0.70**: Preliminary, use with caution

---

## Adding Custom KUs

### Create New KU

```json
{
  "id": "CUSTOM-001",
  "domain": "medicine",
  "invariant": "Your constraint description",
  "applies_if": {
    "parameter_name <": threshold_value
  },
  "failure_modes": ["failure_1", "failure_2"],
  "metadata": {
    "source": "your_source",
    "confidence": 0.90,
    "date_added": "2026-01-05"
  }
}
```

### Add to Library

```python
import json

# Load existing KUs
with open('nanocern_reactor/generated_kus/medical/medical_kus.json', 'r') as f:
    medical_kus = json.load(f)

# Add new KU
new_ku = {
    "id": "CUSTOM-001",
    "domain": "medicine",
    # ... rest of KU definition
}

medical_kus.append(new_ku)

# Save updated library
with open('nanocern_reactor/generated_kus/medical/medical_kus.json', 'w') as f:
    json.dump(medical_kus, f, indent=2)
```

---

## Validation & Testing

### Validate KU Structure

```python
def validate_ku(ku):
    """Validate KU has required fields"""
    required_fields = ['id', 'domain', 'invariant', 'applies_if', 'failure_modes']
    
    for field in required_fields:
        if field not in ku:
            return False, f"Missing field: {field}"
    
    return True, "Valid"

# Test all KUs
for ku in medical_kus:
    valid, message = validate_ku(ku)
    if not valid:
        print(f"Invalid KU {ku.get('id', 'unknown')}: {message}")
```

---

## Research Applications

### Drug Safety Screening

Use medical KUs to screen drug prescriptions against patient state.

### Physics Simulation Validation

Use physics KUs to validate simulation parameters before running expensive computations.

### Chemistry Reaction Safety

Use chemistry KUs to check reaction conditions against safety limits.

### Mathematical Method Selection

Use mathematics KUs to select appropriate numerical methods based on problem characteristics.

---

## Important Notes

1. **Generated KUs require validation** - Always verify against authoritative sources
2. **Confidence scores are estimates** - Use clinical judgment
3. **Not for clinical use** - Research and education only
4. **Update regularly** - Medical knowledge evolves
5. **Cite sources** - Track provenance of all KUs

---

## Future Enhancements

- [ ] Add more FDA/EMA-sourced medical KUs
- [ ] Expand physics KUs to quantum mechanics
- [ ] Include biochemistry KUs
- [ ] Add engineering constraint KUs
- [ ] Implement KU versioning system
- [ ] Create KU validation test suite

---

**Total Knowledge Units**: 231  
**Last Updated**: 2026-01-05  
**Status**: Research Use Only
