"""
Trial: Super-Agers vs Baseline.
The battle for age 100.
"""

from body_state_model import HumanBodyState
from longevity_orchestrator import LongevityOrchestrator

def run_super_ager_trial():
    print("="*80)
    print("GEROSCIENCE TRIAL: THE GENETIC SUPER-AGER FORECAST")
    print("="*80)
    
    # 1. Initialize Baseline Subject
    print("\n[SUBJECT A] Initializing Baseline Twin (Standard Aging)...")
    body_a = HumanBodyState()
    orch_a = LongevityOrchestrator(body_a)
    history_a = orch_a.simulate_lifespan(years=100, genetic_profile=[], verbose=True)
    
    # 2. Initialize Super-Ager Subject
    print("\n[SUBJECT B] Initializing Longevi-Twin (FOXO3 + APOE \u03b52)...")
    body_b = HumanBodyState()
    orch_b = LongevityOrchestrator(body_b)
    history_b = orch_b.simulate_lifespan(
        years=100, 
        genetic_profile=["FOXO3_LONGEVITY", "APOE_E2"], 
        verbose=True
    )
    
    # 3. Final Comparison
    print("\n" + "="*80)
    print("TRIAL VERDICT: HEALTHSPAN TRAJECTORY")
    print("="*80)
    
    age_a = history_a[-1]["age"]
    score_a = history_a[-1]["homeostasis"]
    age_b = history_b[-1]["age"]
    score_b = history_b[-1]["homeostasis"]
    
    print(f"Baseline: Reached age {age_a} with {score_a:.1f}% homeostasis.")
    print(f"  • Brain Neurons: {body_a.brain.neuron_count_billion:.2f}B")
    print(f"  • Kidney GFR:   {body_a.kidneys.gfr_ml_min:.1f} mL/min")
    
    print(f"\nSuper-Ager: Reached age {age_b} with {score_b:.1f}% homeostasis.")
    print(f"  • Brain Neurons: {body_b.brain.neuron_count_billion:.2f}B")
    print(f"  • Kidney GFR:   {body_b.kidneys.gfr_ml_min:.1f} mL/min")
    
    # Calculate Vital Reserve Gain
    neuron_gain = ((body_b.brain.neuron_count_billion - body_a.brain.neuron_count_billion) / body_a.brain.neuron_count_billion) * 100
    gfr_gain = body_b.kidneys.gfr_ml_min - body_a.kidneys.gfr_ml_min
    
    print("\n" + "-"*40)
    print("GENETIC ADVANTAGE ANALYSIS")
    print("-"*40)
    print(f"Neural Preservation: +{neuron_gain:.1f}% higher brain density at age 100.")
    print(f"Renal Reserve:       +{gfr_gain:.1f} mL/min GFR advantage.")
    print("\n[VERDICT] APOE \u03b52 and FOXO3 successfully preserved critical structural invariants.")

if __name__ == "__main__":
    run_super_ager_trial()
