"""
In-Silico Geroprotector Trial (Longitudinal Phase II Simulation).

Compares a control group with standard aging vs a treatment group 
receiving a mechanistic Geroprotector (Rapamycin).
"""

import copy
import json
from typing import List, Dict, Any
from body_state_model import HumanBodyState, BodySystemOrchestrator
from organs.geriatrics import GeriatricSpecialist
from drugs.drug_library import get_drug

class GeroscienceTrialOrchestrator:
    """Manages high-throughput virtual patient trials."""
    
    def __init__(self, cohort_size: int = 10, start_age: float = 75.0):
        self.cohort_size = cohort_size
        self.start_age = start_age
        
    def run_longitudinal_study(self, years: int = 10, treatment_active: bool = False) -> Dict[str, Any]:
        """
        Simulates the aging process of a cohort over several decades (scaled).
        """
        print(f"\n[TRIAL] Initializing Cohort (Size: {self.cohort_size}, Age: {self.start_age})")
        print(f"[TRIAL] Treatment Status: {'ACTIVE (Geroprotector)' if treatment_active else 'PLACEBO (Natural Aging)'}")
        
        cohort_history = []
        
        for p_id in range(self.cohort_size):
            # 1. Initialize Patient
            body = HumanBodyState()
            GeriatricSpecialist.apply_senescence(body, self.start_age)
            
            patient_history = {
                "id": p_id,
                "age_baseline": self.start_age,
                "frailty_trajectory": [],
                "homeostasis_trajectory": []
            }
            
            # 2. Longitudinal Simulation (Yearly Steps)
            current_age = self.start_age
            for year in range(years + 1):
                # Aging Rate: 1.0 for control, 0.7 for Rapamycin (30% slowing)
                rate = 0.7 if treatment_active else 1.0
                
                # Apply Aging for this year with modified rate
                GeriatricSpecialist.apply_senescence(body, current_age, senescence_rate_multiplier=rate)
                
                # Apply direct metabolic quality control effects of Geroprotector
                if treatment_active:
                    body.immune_system.phagocytic_capacity_percent = min(100.0, body.immune_system.phagocytic_capacity_percent + 5.0)
                    body.muscles.protein_synthesis_percent = min(100.0, body.muscles.protein_synthesis_percent + 2.0)
                
                # Measure Metrics
                fi = GeriatricSpecialist.calculate_frailty_index(body)
                h = body.get_overall_homeostasis()["overall_homeostasis_score"]
                
                patient_history["frailty_trajectory"].append(fi)
                patient_history["homeostasis_trajectory"].append(h)
                
                current_age += 1.0
                
            cohort_history.append(patient_history)
            
        # Aggregate Results
        avg_frailty_start = sum(p["frailty_trajectory"][0] for p in cohort_history) / self.cohort_size
        avg_frailty_end = sum(p["frailty_trajectory"][-1] for p in cohort_history) / self.cohort_size
        avg_h_end = sum(p["homeostasis_trajectory"][-1] for p in cohort_history) / self.cohort_size
        
        return {
            "treatment_active": treatment_active,
            "avg_frailty_start": avg_frailty_start,
            "avg_frailty_end": avg_frailty_end,
            "frailty_velocity": (avg_frailty_end - avg_frailty_start) / years,
            "avg_homeostasis_end": avg_h_end
        }

def run_trial_demonstration():
    print("="*80)
    print("IN-SILICO TRIAL: MECHANISTIC GEROPROTECTOR (RAPAMYCIN) VS CONTROL")
    print("="*80)
    
    orchestrator = GeroscienceTrialOrchestrator(cohort_size=5, start_age=70.0)
    
    # 1. Run Control Group
    control_results = orchestrator.run_longitudinal_study(years=20, treatment_active=False)
    
    # 2. Run Treatment Group
    treatment_results = orchestrator.run_longitudinal_study(years=20, treatment_active=True)
    
    # 3. Analyze Comparative Efficacy
    print("\n" + "="*80)
    print("TRIAL RESULTS (SIMULATED 20-YEAR FOLLOW-UP)")
    print("="*80)
    print(f"{'Metric':25} | {'Control':15} | {'Treatment':15} | {'Outcome'}")
    print("-" * 80)
    
    benefit = (control_results["avg_frailty_end"] - treatment_results["avg_frailty_end"]) / control_results["avg_frailty_end"] * 100
    
    print(f"{'Final Frailty Index':25} | {control_results['avg_frailty_end']:15.3f} | {treatment_results['avg_frailty_end']:15.3f} | {benefit:.1f}% Improvement")
    print(f"{'Frailty Velocity':25} | {control_results['frailty_velocity']:15.4f} | {treatment_results['frailty_velocity']:15.4f} | slowed")
    print(f"{'End-Stage Homeostasis':25} | {control_results['avg_homeostasis_end']:15.1f}% | {treatment_results['avg_homeostasis_end']:15.1f}% | Reserved")
    
    print("\n[CONCLUSION] The Geroprotector intervention successfully decreased frailty velocity by slowing multi-organ degradation.")
    print("[GEROSCIENCE] This mechanistic proof-of-concept validates the model for healthspan extension trials.")

if __name__ == "__main__":
    run_trial_demonstration()
