"""
Demonstration scenarios for the Complete State-Based Physiological Model.

Shows various disease progressions and multi-organ cascades.
"""

from body_state_model import HumanBodyState, BodySystemOrchestrator
import json


def demo_1_healthy_aging():
    """Simulate healthy aging process."""
    print("=" * 70)
    print("DEMO 1: Healthy Aging (Baseline)")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\nStarting state (age 30):")
    print(body.get_state_summary())
    
    # Simulate 60 seconds of normal physiology
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nFinal state:")
    print(body.get_state_summary())
    
    history = orchestrator.get_history_summary()
    print(f"\nHomeostasis change: {history['initial_homeostasis']:.1f}% → {history['final_homeostasis']:.1f}%")


def demo_2_heart_failure_cascade():
    """Simulate heart failure and observe multi-organ cascade."""
    print("\n" + "=" * 70)
    print("DEMO 2: Heart Failure → Multi-Organ Cascade")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING HEART FAILURE (ischemic cardiomyopathy)...")
    body.heart.transition_to_heart_failure("ischemic")
    
    print("\nImmediate cardiac changes:")
    print(f"  • Ejection fraction: {body.heart.ejection_fraction_percent:.1f}% (normal: 60%)")
    print(f"  • Cardiac output: {body.heart.cardiac_output_l_min:.2f} L/min (normal: 5.0)")
    print(f"  • BNP: {body.heart.bnp_pg_ml:.0f} pg/mL (normal: 50)")
    
    # Run simulation to observe cascade
    print("\n>>> Running 60-second simulation to observe cascade effects...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nCascade effects on other organs:")
    print(f"  • Kidney GFR: {body.kidneys.gfr_ml_min:.1f} mL/min (normal: 100)")
    print(f"  • Kidney fluid retention: {body.kidneys.fluid_retention_ml:.0f} mL")
    print(f"  • Brain perfusion: {body.brain.cerebral_blood_flow_ml_min:.0f} mL/min (normal: 750)")
    print(f"  • Liver perfusion: {body.liver.hepatic_blood_flow_ml_min:.0f} mL/min (normal: 1500)")
    
    print("\nOverall body state:")
    homeostasis = body.get_overall_homeostasis()
    print(f"  • Overall homeostasis: {homeostasis['overall_homeostasis_score']:.1f}%")
    print(f"  • Imbalanced organs: {', '.join(homeostasis['imbalanced_organs'])}")


def demo_3_copd_hypoxia_cascade():
    """Simulate COPD causing hypoxia and affecting heart/brain."""
    print("\n" + "=" * 70)
    print("DEMO 3: COPD → Hypoxia → Cardiac/Neurological Effects")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING SEVERE COPD...")
    body.lungs.transition_to_copd("severe")
    
    print("\nPulmonary changes:")
    print(f"  • FEV1: {body.lungs.fev1_percent_predicted:.1f}% predicted")
    print(f"  • Arterial PO2: {body.lungs.arterial_po2_mmhg:.1f} mmHg (normal: 95)")
    print(f"  • Arterial PCO2: {body.lungs.arterial_pco2_mmhg:.1f} mmHg (normal: 40)")
    print(f"  • O2 saturation: {body.lungs.arterial_oxygen_saturation_percent:.1f}%")
    
    # Run simulation
    print("\n>>> Running simulation...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nHypoxia effects on other organs:")
    print(f"  • Heart rate: {body.heart.heart_rate_bpm:.1f} bpm (compensatory tachycardia)")
    print(f"  • Cardiac contractility: {body.heart.contractility_percent:.1f}%")
    print(f"  • Brain oxygen delivery: {body.brain.oxygen_consumption_ml_min:.1f} mL/min")
    print(f"  • Myocardial oxygen delivery: {body.heart.myocardial_oxygen_delivery_percent:.1f}%")
    
    print("\nOverall body state:")
    print(body.get_state_summary())


def demo_4_cirrhosis_multisystem_failure():
    """Simulate liver cirrhosis affecting multiple systems."""
    print("\n" + "=" * 70)
    print("DEMO 4: Liver Cirrhosis → Multisystem Effects")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING LIVER CIRRHOSIS (Child-Pugh C - severe)...")
    body.liver.transition_to_cirrhosis("C")
    
    print("\nHepatic dysfunction:")
    print(f"  • Hepatocyte count: {body.liver.hepatocyte_count_billion:.1f} billion (normal: 240)")
    print(f"  • Albumin: {body.liver.albumin_g_dl:.1f} g/dL (normal: 4.0)")
    print(f"  • INR: {body.liver.inr:.1f} (normal: 1.0)")
    print(f"  • Ammonia clearance: {body.liver.ammonia_clearance_percent:.1f}%")
    print(f"  • Bilirubin: {body.liver.bilirubin_mg_dl:.1f} mg/dL (normal: 0.8)")
    
    # Run simulation
    print("\n>>> Running simulation...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nMultisystem effects:")
    print(f"  • Brain ammonia toxicity: {body.brain.ammonia_toxicity:.2f} (hepatic encephalopathy)")
    print(f"  • Brain consciousness: {body.brain.consciousness_level}")
    print(f"  • Liver bile production: {body.liver.bile_production_ml_day:.0f} mL/day")
    
    print("\nOverall body state:")
    print(body.get_state_summary())


def demo_5_ckd_anemia_bone_disease():
    """Simulate chronic kidney disease affecting blood and bones."""
    print("\n" + "=" * 70)
    print("DEMO 5: Chronic Kidney Disease → Anemia + Bone Disease")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING CKD STAGE 4 (severe)...")
    body.kidneys.transition_to_ckd(stage=4)
    
    print("\nRenal dysfunction:")
    print(f"  • GFR: {body.kidneys.gfr_ml_min:.1f} mL/min (normal: 100)")
    print(f"  • Creatinine: {body.kidneys.serum_creatinine_mg_dl:.1f} mg/dL (normal: 1.0)")
    print(f"  • Erythropoietin: {body.kidneys.erythropoietin_mu_ml:.1f} mU/mL (normal: 10)")
    print(f"  • Vitamin D activation: {body.kidneys.vitamin_d_activation_percent:.1f}%")
    print(f"  • Fluid retention: {body.kidneys.fluid_retention_ml:.0f} mL")
    
    # Run simulation
    print("\n>>> Running simulation...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nSecondary effects:")
    print(f"  • Heart preload (from fluid): {body.heart.preload_mmhg:.1f} mmHg")
    print(f"  • Heart BNP: {body.heart.bnp_pg_ml:.1f} pg/mL")
    print("  • Expected anemia (low EPO)")
    print("  • Expected bone disease (low vitamin D)")
    
    print("\nOverall body state:")
    print(body.get_state_summary())


def demo_6_type1_diabetes_dka():
    """Simulate Type 1 diabetes with severe hyperglycemia."""
    print("\n" + "=" * 70)
    print("DEMO 6: Type 1 Diabetes → Severe Hyperglycemia")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING TYPE 1 DIABETES (90% beta cell destruction)...")
    body.pancreas.transition_to_type1_diabetes(autoimmune_destruction_percent=90.0)
    
    print("\nPancreatic dysfunction:")
    print(f"  • Beta cell mass: {body.pancreas.beta_cell_mass_mg:.1f} mg (normal: 1000)")
    print(f"  • Insulin secretion: {body.pancreas.insulin_secretion_mu_ml:.1f} μU/mL (normal: 10)")
    print(f"  • Glucagon: {body.pancreas.glucagon_secretion_pg_ml:.1f} pg/mL (normal: 50)")
    print(f"  • Fasting glucose: {body.pancreas.fasting_glucose_mg_dl:.0f} mg/dL (normal: 90)")
    print(f"  • HbA1c: {body.pancreas.hba1c_percent:.1f}% (normal: 5.0)")
    
    # Run simulation
    print("\n>>> Running simulation...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nMetabolic effects:")
    print(f"  • Liver glucose production: {body.liver.glucose_production_mg_dl_hr:.0f} mg/dL/hr")
    print("  • Expected muscle glucose uptake impairment")
    print("  • Risk of diabetic ketoacidosis (DKA)")
    
    print("\nOverall body state:")
    print(body.get_state_summary())


def demo_7_stroke_acute_brain_injury():
    """Simulate acute stroke."""
    print("\n" + "=" * 70)
    print("DEMO 7: Acute Stroke → Brain Injury")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING ACUTE STROKE (100 mL infarct)...")
    body.brain.transition_to_stroke(affected_volume_ml=100.0)
    
    print("\nNeurological damage:")
    print(f"  • Neuron count: {body.brain.neuron_count_billion:.1f} billion (normal: 86)")
    print(f"  • Cerebral blood flow: {body.brain.cerebral_blood_flow_ml_min:.0f} mL/min (normal: 750)")
    print(f"  • ATP production: {body.brain.atp_production_percent:.1f}%")
    print(f"  • Consciousness: {body.brain.consciousness_level}")
    print(f"  • Consciousness impairment: {body.brain.consciousness_impairment:.2f}")
    
    # Run simulation
    print("\n>>> Running simulation...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nOverall body state:")
    print(body.get_state_summary())


def demo_8_alzheimers_progression():
    """Simulate Alzheimer's disease progression."""
    print("\n" + "=" * 70)
    print("DEMO 8: Alzheimer's Disease → Cognitive Decline")
    print("=" * 70)
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("\n>>> INDUCING MODERATE ALZHEIMER'S DISEASE...")
    body.brain.transition_to_alzheimers("moderate")
    
    print("\nNeurodegenerative changes:")
    print(f"  • Neuron count: {body.brain.neuron_count_billion:.1f} billion (normal: 86)")
    print(f"  • Synapse count: {body.brain.synapse_count_trillion:.1f} trillion (normal: 100)")
    print(f"  • Acetylcholine: {body.brain.acetylcholine_nmol_g:.2f} nmol/g (normal: 0.30)")
    print(f"  • BDNF: {body.brain.bdnf_ng_ml:.1f} ng/mL (normal: 25)")
    print(f"  • Mitochondrial function: {body.brain.mitochondrial_function_percent:.1f}%")
    
    print("\nCognitive impairment:")
    print(f"  • Memory score: {body.brain.memory_score:.1f}/100")
    print(f"  • Processing speed: {body.brain.processing_speed_ms:.0f} ms (normal: 200)")
    print(f"  • Executive function: {body.brain.executive_function_score:.1f}/100")
    
    # Run simulation
    print("\n>>> Running simulation...")
    orchestrator.run_simulation(60.0, verbose=False)
    
    print("\nOverall body state:")
    print(body.get_state_summary())


def run_all_demos():
    """Run all demonstration scenarios."""
    demo_1_healthy_aging()
    demo_2_heart_failure_cascade()
    demo_3_copd_hypoxia_cascade()
    demo_4_cirrhosis_multisystem_failure()
    demo_5_ckd_anemia_bone_disease()
    demo_6_type1_diabetes_dka()
    demo_7_stroke_acute_brain_injury()
    demo_8_alzheimers_progression()
    
    print("\n" + "=" * 70)
    print("ALL DEMONSTRATIONS COMPLETE")
    print("=" * 70)


if __name__ == "__main__":
    run_all_demos()
