"""
Geriatric Medicine Demonstration.

Focuses on:
1. Aging-related physiological decline (85-year-old patient).
2. Frailty assessment.
3. Polypharmacy risks and Beers Criteria validation.
4. Multi-system failure in the elderly.
"""

import time
from body_state_model import HumanBodyState, BodySystemOrchestrator
from drugs import get_drug, list_available_drugs
from organs.geriatrics import GeriatricSpecialist
from safety_reactor import SafetyReactor

def run_geriatric_demo():
    print("="*80)
    print("GERIATRIC MEDICINE SIMULATION: 85-Year-Old Female Patient")
    print("="*80)
    
    # 1. Initialize Patient
    body = HumanBodyState()
    age = 85.0
    print(f"\n[INIT] Applying Senescence for Age: {age}")
    GeriatricSpecialist.apply_senescence(body, age)
    
    # 2. Assess Basel Geriatric State
    frailty = GeriatricSpecialist.calculate_frailty_index(body)
    print(f"[RESERVE] Physiological Reserve Assessment:")
    print(f"  • Frailty Index: {frailty:.2f} ({'FRAIL' if frailty > 0.25 else 'ROBUST'})")
    print(f"  • Kidney GFR: {body.kidneys.gfr_ml_min:.1f} mL/min (Age-related decline)")
    print(f"  • Muscle Mass: {body.muscles.muscle_mass_kg:.1f} kg")
    print(f"  • Brain Volume: {(body.brain.neuron_count_billion/86.0)*100:.1f}% of youthful baseline")
    print(f"  • Heart Afterload: {body.heart.afterload_mmhg:.1f} mmHg (Increased stiffness)")

    # 3. Polypharmacy Safety Validation (Medical NanoCERN)
    reactor = SafetyReactor()
    print(f"\n[PHARMA] Evaluating Polypharmacy Plan (Medical NanoCERN):")
    
    drugs_to_test = [
        ("Lisinopril", 20.0),      # For hypertension
        ("Metformin", 1000.0),     # For diabetes
        ("Lorazepam", 1.0),       # Added for "anxiety" (Geriatric PIM)
        ("Ibuprofen", 400.0)       # Added for "joint pain" (Geriatric PIM + CKD risk)
    ]
    
    for drug_name, dose in drugs_to_test:
        drug = get_drug(drug_name, dose)
        result = reactor.validate_drug(drug, body)
        
        status = "✅ SAFE" if result["is_safe"] else "❌ REJECTED"
        print(f"\nEvaluating: {drug_name} {dose}mg | Status: {status}")
        
        if not result["is_safe"]:
            for v in result["violations"]:
                print(f"  • {v['type']}: {v.get('reason', v.get('parameter'))}")
                if 'severity' in v:
                    print(f"    Severity: {v['severity'].upper()}")

    # 4. Challenge Scenario: Acute Delirium & Multi-System Decompensation
    print("\n" + "="*80)
    print("SCENARIO: Acute Infection triggering Multi-System Failure (Frailty Cascade)")
    print("="*80)
    
    # Trigger immune system failure / sepsis-like state
    body.immune_system.transition_to_immunodeficiency()
    body.immune_system.innate_immune_reserve_percent = 20.0
    
    orchestrator = BodySystemOrchestrator(body)
    
    print("\nSimulation Loop (Time-Steps):")
    for t in range(5):
        # Cascading failure: Immune -> Brain (Delirium) -> Heart -> Kidneys
        if t == 1:
            print("  [t=1] Sepsis cascade triggering Brain Decompensation (Delirium)")
            body.brain.processing_speed_ms += 200.0
            body.brain.executive_function_score -= 50.0
            body.brain.state_classification = "DELIRIUM"
            
        if t == 2:
            print("  [t=2] Heart failing under metabolic stress")
            body.heart.transition_to_heart_failure(cause="hypertensive")
            
        orchestrator.step()
        
        homeostasis_data = body.get_overall_homeostasis()
        homeostasis = homeostasis_data["overall_homeostasis_score"]
        critical_organs = [o for o, s in homeostasis_data["organ_scores"].items() if s < 50]
        
        print(f"  Week {t+1}: Homeostasis {homeostasis:.1f}% | Critical: {critical_organs}")
        
        if homeostasis < 50:
            print("  ⚠️ CRITICAL MULTI-SYSTEM FAILURE DETECTED")
            break
            
    # 5. Summary
    print("\n" + "="*80)
    print("GERIATRIC MEDICINE CONCLUSIONS")
    print("="*80)
    final_frailty = GeriatricSpecialist.calculate_frailty_index(body)
    print(f"Final Patient State:")
    print(f"  • Condition: {body.brain.state_classification} + Heart Failure")
    print(f"  • Terminal Frailty Index: {final_frailty:.2f}")
    print(f"  • Clinical Recommendation: Transition to palliative/comfort care envelope.")

if __name__ == "__main__":
    run_geriatric_demo()
