"""
Drug Intervention Demonstrations.

Shows complete drug discovery workflow:
1. Model disease state
2. Propose drug intervention
3. Validate safety (Medical NanoCERN)
4. Simulate effects
5. Measure therapeutic benefit
"""

from body_state_model import HumanBodyState, BodySystemOrchestrator
from drugs import (
    create_ace_inhibitor,
    create_beta_blocker,
    create_metformin,
    create_coq10_supplement,
    validate_drug_safety
)


def demo_1_ace_inhibitor_for_heart_failure():
    """Demonstrate ACE inhibitor therapy for heart failure."""
    print("=" * 70)
    print("DEMO 1: ACE Inhibitor for Heart Failure")
    print("=" * 70)
    
    # Create body with heart failure
    body = HumanBodyState()
    body.heart.transition_to_heart_failure("ischemic")
    
    print("\n>>> BASELINE (Heart Failure):")
    print(f"  • Ejection fraction: {body.heart.ejection_fraction_percent:.1f}%")
    print(f"  • Cardiac output: {body.heart.cardiac_output_l_min:.2f} L/min")
    print(f"  • Afterload: {body.heart.afterload_mmhg:.1f} mmHg")
    print(f"  • BNP: {body.heart.bnp_pg_ml:.0f} pg/mL")
    
    # Create drug intervention
    drug = create_ace_inhibitor(dose_mg=10.0)
    
    # Validate safety
    print("\n>>> SAFETY VALIDATION (Medical NanoCERN):")
    safety = validate_drug_safety(drug, body)
    if safety["is_safe"]:
        print("  ✅ SAFE - No contraindications detected")
    else:
        print("  ❌ UNSAFE - Contraindications:")
        for contra in safety["contraindications"]:
            print(f"    • {contra['reason']}")
        return
    
    # Simulate drug administration
    print("\n>>> ADMINISTERING ACE INHIBITOR (10 mg oral)...")
    
    orchestrator = BodySystemOrchestrator(body)
    
    # Simulate 24 hours with drug
    for hour in range(24):
        drug.update_state(time_step_hr=1.0)
        
        if drug.is_therapeutic():
            effects = drug.apply_to_body(body)
        
        # Run body simulation for 1 hour (3600 seconds)
        for _ in range(36):
            orchestrator.step()
    
    print(f"\n>>> AFTER 24 HOURS:")
    print(f"  • Drug concentration: {drug.plasma_concentration_mg_l:.2f} mg/L")
    print(f"  • Therapeutic range: {drug.is_therapeutic()}")
    print(f"  • Afterload: {body.heart.afterload_mmhg:.1f} mmHg (↓ from baseline)")
    print(f"  • Cardiac output: {body.heart.cardiac_output_l_min:.2f} L/min")
    print(f"  • BNP: {body.heart.bnp_pg_ml:.0f} pg/mL")


def demo_2_metformin_for_diabetes():
    """Demonstrate metformin therapy for Type 2 diabetes."""
    print("\n" + "=" * 70)
    print("DEMO 2: Metformin for Type 2 Diabetes")
    print("=" * 70)
    
    # Create body with Type 2 diabetes
    body = HumanBodyState()
    body.pancreas.transition_to_type2_diabetes(insulin_resistance_percent=70.0)
    
    print("\n>>> BASELINE (Type 2 Diabetes):")
    print(f"  • Fasting glucose: {body.pancreas.fasting_glucose_mg_dl:.0f} mg/dL")
    print(f"  • HbA1c: {body.pancreas.hba1c_percent:.1f}%")
    print(f"  • Insulin sensitivity: {body.pancreas.insulin_sensitivity_index:.1f}%")
    print(f"  • Liver glucose production: {body.liver.glucose_production_mg_dl_hr:.0f} mg/dL/hr")
    
    # Create drug intervention
    drug = create_metformin(dose_mg=1000.0)
    
    # Validate safety
    print("\n>>> SAFETY VALIDATION:")
    safety = validate_drug_safety(drug, body)
    if safety["is_safe"]:
        print("  ✅ SAFE - No contraindications")
    else:
        print("  ❌ UNSAFE:")
        for contra in safety["contraindications"]:
            print(f"    • {contra['reason']}")
        return
    
    print("\n>>> ADMINISTERING METFORMIN (1000 mg oral)...")
    
    orchestrator = BodySystemOrchestrator(body)
    
    # Simulate 12 hours
    for hour in range(12):
        drug.update_state(time_step_hr=1.0)
        
        if drug.is_therapeutic():
            effects = drug.apply_to_body(body)
        
        for _ in range(36):
            orchestrator.step()
    
    print(f"\n>>> AFTER 12 HOURS:")
    print(f"  • Drug concentration: {drug.plasma_concentration_mg_l:.2f} mg/L")
    print(f"  • Liver glucose production: {body.liver.glucose_production_mg_dl_hr:.0f} mg/dL/hr (↓)")
    print(f"  • Expected glucose reduction: ~30-50 mg/dL")


def demo_3_coq10_for_mitochondrial_support():
    """Demonstrate CoQ10 supplementation for mitochondrial support."""
    print("\n" + "=" * 70)
    print("DEMO 3: CoQ10 for Mitochondrial Support (Heart Failure)")
    print("=" * 70)
    
    # Create body with heart failure
    body = HumanBodyState()
    body.heart.transition_to_heart_failure("ischemic")
    
    print("\n>>> BASELINE:")
    print(f"  • Heart ATP production: {body.heart.atp_production_percent:.1f}%")
    print(f"  • Heart mitochondrial density: {body.heart.mitochondrial_density_percent:.1f}%")
    print(f"  • Brain mitochondrial function: {body.brain.mitochondrial_function_percent:.1f}%")
    
    # Create drug intervention
    drug = create_coq10_supplement(dose_mg=200.0)
    
    print("\n>>> ADMINISTERING CoQ10 (200 mg oral)...")
    
    orchestrator = BodySystemOrchestrator(body)
    
    # Simulate 7 days (chronic supplementation)
    for day in range(7):
        # Daily dose
        drug.update_state(time_step_hr=24.0)
        
        if drug.is_therapeutic():
            effects = drug.apply_to_body(body)
        
        # Simulate day
        for _ in range(240):  # 24 hours
            orchestrator.step()
    
    print(f"\n>>> AFTER 7 DAYS:")
    print(f"  • Heart ATP production: {body.heart.atp_production_percent:.1f}% (↑)")
    print(f"  • Heart mitochondrial density: {body.heart.mitochondrial_density_percent:.1f}% (↑)")
    print(f"  • Brain mitochondrial function: {body.brain.mitochondrial_function_percent:.1f}% (↑)")


def demo_4_contraindication_detection():
    """Demonstrate Medical NanoCERN contraindication detection."""
    print("\n" + "=" * 70)
    print("DEMO 4: Contraindication Detection (Medical NanoCERN)")
    print("=" * 70)
    
    # Create body with severe renal impairment
    body = HumanBodyState()
    body.kidneys.transition_to_ckd(stage=4)
    
    print("\n>>> PATIENT STATE:")
    print(f"  • Kidney GFR: {body.kidneys.gfr_ml_min:.1f} mL/min (CKD Stage 4)")
    print(f"  • Creatinine: {body.kidneys.serum_creatinine_mg_dl:.1f} mg/dL")
    
    # Try to prescribe metformin (contraindicated)
    drug = create_metformin(dose_mg=1000.0)
    
    print("\n>>> ATTEMPTING TO PRESCRIBE METFORMIN...")
    
    safety = validate_drug_safety(drug, body)
    
    if safety["is_safe"]:
        print("  ✅ SAFE")
    else:
        print("  ❌ UNSAFE - PRESCRIPTION REJECTED")
        print("\n  Contraindications detected:")
        for contra in safety["contraindications"]:
            print(f"\n  • Type: {contra['type']}")
            print(f"    Organ: {contra['organ']}")
            print(f"    Parameter: {contra['parameter']} = {contra['value']:.1f}")
            print(f"    Threshold: {contra['threshold']:.1f}")
            print(f"    Severity: {contra['severity']}")
            print(f"    Reason: {contra['reason']}")
    
    print("\n>>> RESULT: Prescription prevented before harm occurs")


def demo_5_beta_blocker_bradycardia():
    """Demonstrate beta blocker contraindication in bradycardia."""
    print("\n" + "=" * 70)
    print("DEMO 5: Beta Blocker Contraindication (Bradycardia)")
    print("=" * 70)
    
    # Create body with heart block (bradycardia)
    body = HumanBodyState()
    body.heart.transition_to_arrhythmia("heart_block")
    
    print("\n>>> PATIENT STATE:")
    print(f"  • Heart rate: {body.heart.heart_rate_bpm:.1f} bpm (bradycardia)")
    print(f"  • PR interval: {body.heart.pr_interval_ms:.0f} ms (prolonged)")
    
    # Try to prescribe beta blocker (contraindicated)
    drug = create_beta_blocker(dose_mg=50.0)
    
    print("\n>>> ATTEMPTING TO PRESCRIBE BETA BLOCKER...")
    
    safety = validate_drug_safety(drug, body)
    
    if safety["is_safe"]:
        print("  ✅ SAFE")
    else:
        print("  ❌ UNSAFE - PRESCRIPTION REJECTED")
        for contra in safety["contraindications"]:
            print(f"\n  • {contra['reason']}")
            print(f"    Current HR: {contra['value']:.1f} bpm")
            print(f"    Minimum safe HR: {contra['threshold']:.1f} bpm")
            print(f"    Severity: {contra['severity']}")
    
    print("\n>>> RESULT: Prevented potentially fatal bradycardia")


def run_all_drug_demos():
    """Run all drug intervention demonstrations."""
    demo_1_ace_inhibitor_for_heart_failure()
    demo_2_metformin_for_diabetes()
    demo_3_coq10_for_mitochondrial_support()
    demo_4_contraindication_detection()
    demo_5_beta_blocker_bradycardia()
    
    print("\n" + "=" * 70)
    print("ALL DRUG INTERVENTION DEMONSTRATIONS COMPLETE")
    print("=" * 70)
    print("\n✅ Drug Discovery Workflow Validated:")
    print("  1. Disease modeling ✓")
    print("  2. Drug intervention ✓")
    print("  3. Safety validation (Medical NanoCERN) ✓")
    print("  4. Pharmacokinetic simulation ✓")
    print("  5. Therapeutic effect measurement ✓")
    print("  6. Contraindication detection ✓")


if __name__ == "__main__":
    run_all_drug_demos()
