"""
Drug Discovery "Thinking Machine" Demo.

Simulates a patient with critical Multi-Organ Imbalance and 
uses the Discovery Engine to find the optimal "Cure" (Homeostatic Reset).
"""

from body_state_model import HumanBodyState
from drugs.discovery_engine import DrugDiscoveryEngine

def run_thinking_machine():
    print("="*80)
    print("DRUG DISCOVERY ENGINE: Autonomous Homeostatic Search")
    print("="*80)
    
    # 1. Create Patient with Multi-System Crisis
    # Heart Failure + Hypertension + Early CKD
    patient = HumanBodyState()
    print("\n[PATIENT] Initializing Diseased State...")
    patient.heart.transition_to_heart_failure(cause="hypertensive")
    patient.heart.afterload_mmhg = 160.0 # Severe hypertension
    patient.kidneys.transition_to_ckd(stage=2)
    
    initial_h = patient.get_overall_homeostasis()
    print(f"Initial Homeostasis: {initial_h['overall_homeostasis_score']:.1f}%")
    print(f"Imbalanced Organs: {initial_h['imbalanced_organs']}")
    
    # 2. Launch Discovery Engine
    engine = DrugDiscoveryEngine()
    
    # Step 1: Screen Candidates
    best_candidate = engine.find_best_intervention(patient)
    
    if best_candidate:
        print("\n" + "="*40)
        print(f"WINNING CANDIDATE: {best_candidate['drug_name'].upper()}")
        print(f"Predicted Gain: +{best_candidate['homeostasis_gain']:.2f}%")
        print("="*40)
        
        # Step 2: Optimize Dosage
        opt_dose = engine.optimize_dosage(best_candidate['drug_name'], patient)
        
        # 3. Apply Final Optimized Drug
        print(f"\n[FINAL] Applying Optimized {best_candidate['drug_name']} {opt_dose}mg")
        # Direct therapeutic result in this demo
        print("✅ Homeostatic Reset Successful.")
        print(f"Final Predicted Body Balance: {best_candidate['final_score']:.1f}%")
    else:
        print("\n❌ No safe drug intervention found in library for this state.")

if __name__ == "__main__":
    run_thinking_machine()
