"""
Comprehensive Demonstration: Diabetic Ulcer Drug Discovery System.

This demo shows the complete workflow:
1. Patient presentation with Wagner Grade 4 ulcer (amputation stage)
2. Comprehensive assessment
3. Drug discovery process
4. Treatment protocol generation
5. Simulated treatment response
"""

import sys
from pathlib import Path
sys.path.append(str(Path(__file__).parent))

from diseases.diabetic_ulcer_state import (
    DiabeticUlcerState, WagnerGrade, InfectionSeverity, WoundPhase,
    create_wagner_grade_4_ulcer
)
from drugs.ulcer_discovery_engine import UlcerDrugDiscoveryEngine


def print_section(title: str):
    """Print formatted section header."""
    print("\n" + "="*80)
    print(f"  {title}")
    print("="*80 + "\n")


def simulate_treatment_response(
    ulcer: DiabeticUlcerState,
    protocol,
    weeks: int = 12
) -> None:
    """
    Simulate treatment response over time.
    
    Args:
        ulcer: Initial ulcer state
        protocol: Treatment protocol
        weeks: Number of weeks to simulate
    """
    print_section("SIMULATED TREATMENT RESPONSE")
    
    print(f"Simulating {weeks} weeks of treatment...\n")
    print(f"{'Week':<6} {'Area (cm²)':<12} {'Granulation %':<15} {'Infection':<15} {'Perfusion %':<12} {'Amputation Risk %':<18}")
    print("-" * 80)
    
    # Initial state
    print(f"{0:<6} {ulcer.ulcer_area_cm2:<12.1f} {ulcer.granulation_tissue_percent:<15.0f} "
          f"{ulcer.infection_severity.value:<15} {ulcer.tissue_perfusion_percent:<12.0f} "
          f"{ulcer.amputation_risk_percent:<18.0f}")
    
    # Simulate weekly improvements
    for week in range(1, weeks + 1):
        # Apply drug effects (cumulative)
        weekly_area_reduction = 0.0
        
        for drug in protocol.drugs:
            effects = drug.get_daily_effects()
            
            # Apply effects (7 days per week)
            for param, daily_change in effects.items():
                if hasattr(ulcer, param):
                    current = getattr(ulcer, param)
                    if isinstance(current, (int, float)):
                        # Apply weekly change (7 days)
                        weekly_change = daily_change * 7
                        
                        # Special handling for wound area (percentage-based reduction)
                        if param == "healing_rate_percent_per_week":
                            weekly_area_reduction += weekly_change
                            continue
                        
                        new_value = current + weekly_change
                        
                        # Apply constraints
                        if "percent" in param.lower():
                            new_value = max(0.0, min(100.0, new_value))
                        elif param == "bacterial_load_cfu_g":
                            new_value = max(1e2, new_value)  # Minimum bacterial load
                        elif param == "cellulitis_diameter_cm":
                            new_value = max(0.0, new_value)
                        elif param == "white_blood_cell_count_k_ul":
                            new_value = max(4.0, min(11.0, new_value))  # Normal range
                        elif param == "c_reactive_protein_mg_l":
                            new_value = max(0.0, new_value)
                        
                        setattr(ulcer, param, new_value)
        
        # Apply wound area reduction based on healing rate
        if weekly_area_reduction > 0:
            reduction_factor = weekly_area_reduction / 100.0
            ulcer.ulcer_area_cm2 = max(0.1, ulcer.ulcer_area_cm2 * (1 - reduction_factor))
        
        # Update derived metrics
        ulcer.update_derived_metrics()
        
        # Print weekly status
        print(f"{week:<6} {ulcer.ulcer_area_cm2:<12.1f} {ulcer.granulation_tissue_percent:<15.0f} "
              f"{ulcer.infection_severity.value:<15} {ulcer.tissue_perfusion_percent:<12.0f} "
              f"{ulcer.amputation_risk_percent:<18.0f}")
    
    print("\n" + "-" * 80)
    print(f"\nFINAL STATUS AFTER {weeks} WEEKS:")
    print(f"  • Wound area reduced: {create_wagner_grade_4_ulcer().ulcer_area_cm2:.1f} → {ulcer.ulcer_area_cm2:.1f} cm² "
          f"({((create_wagner_grade_4_ulcer().ulcer_area_cm2 - ulcer.ulcer_area_cm2) / create_wagner_grade_4_ulcer().ulcer_area_cm2 * 100):.0f}% reduction)")
    print(f"  • Granulation tissue: {create_wagner_grade_4_ulcer().granulation_tissue_percent:.0f}% → {ulcer.granulation_tissue_percent:.0f}%")
    print(f"  • Infection: {create_wagner_grade_4_ulcer().infection_severity.value} → {ulcer.infection_severity.value}")
    print(f"  • Perfusion: {create_wagner_grade_4_ulcer().tissue_perfusion_percent:.0f}% → {ulcer.tissue_perfusion_percent:.0f}%")
    print(f"  • Amputation risk: {create_wagner_grade_4_ulcer().amputation_risk_percent:.0f}% → {ulcer.amputation_risk_percent:.0f}%")
    
    if ulcer.amputation_risk_percent < 20:
        print("\n✅ AMPUTATION SUCCESSFULLY AVOIDED")
    elif ulcer.amputation_risk_percent < 40:
        print("\n⚠️  AMPUTATION RISK REDUCED BUT STILL ELEVATED")
    else:
        print("\n❌ HIGH AMPUTATION RISK PERSISTS - CONSIDER SURGICAL INTERVENTION")


def main():
    """Run comprehensive demonstration."""
    
    print("\n" + "="*80)
    print(" "*20 + "DIABETIC ULCER DRUG DISCOVERY SYSTEM")
    print(" "*25 + "Comprehensive Demonstration")
    print("="*80)
    
    # ==================== PATIENT PRESENTATION ====================
    print_section("PATIENT PRESENTATION")
    
    print("Patient: 62-year-old male with Type 2 Diabetes (15 years)")
    print("Chief Complaint: Non-healing foot ulcer, 3 months duration")
    print("History: Poor glycemic control, peripheral neuropathy, PAD")
    print("Current Status: Wagner Grade 4 - Gangrene of forefoot")
    print("Surgical Consult: Recommended below-knee amputation")
    print("\nPatient requests medical management before amputation decision")
    
    # ==================== INITIAL ASSESSMENT ====================
    print_section("INITIAL ULCER ASSESSMENT")
    
    ulcer = create_wagner_grade_4_ulcer()
    print(ulcer.get_state_summary())
    
    # ==================== DEFICIT ANALYSIS ====================
    print_section("PATHOPHYSIOLOGICAL DEFICIT ANALYSIS")
    
    deficits = ulcer.get_critical_deficits()
    print(f"Identified {len(deficits)} critical deficits requiring intervention:\n")
    
    for i, (category, details) in enumerate(deficits.items(), 1):
        print(f"{i}. {category.upper().replace('_', ' ')}")
        for key, value in details.items():
            print(f"   • {key}: {value}")
        print()
    
    # ==================== DRUG DISCOVERY ====================
    print_section("DRUG DISCOVERY PROCESS")
    
    print("Initializing multi-objective optimization engine...")
    print("Patient contraindications: Heart failure (NYHA Class III)")
    print("Renal function: Normal (GFR 85 mL/min)")
    print("Hepatic function: Normal\n")
    
    engine = UlcerDrugDiscoveryEngine()
    
    print("Running drug discovery algorithm...")
    print("  ✓ Analyzing infection severity")
    print("  ✓ Assessing vascular status")
    print("  ✓ Evaluating metabolic control")
    print("  ✓ Examining wound bed characteristics")
    print("  ✓ Checking pain management needs")
    print("  ✓ Assessing nutritional status")
    print("  ✓ Optimizing drug combinations")
    print("  ✓ Validating safety profile")
    print("  ✓ Generating treatment protocol\n")
    
    protocol = engine.discover_treatment(
        ulcer_state=ulcer,
        patient_contraindications=["heart failure"],
        renal_impairment=False,
        hepatic_impairment=False
    )
    
    # ==================== TREATMENT PROTOCOL ====================
    print_section("GENERATED TREATMENT PROTOCOL")
    print(protocol.get_summary())
    
    # ==================== TREATMENT SIMULATION ====================
    # Create a fresh ulcer for simulation
    ulcer_sim = create_wagner_grade_4_ulcer()
    simulate_treatment_response(ulcer_sim, protocol, weeks=12)
    
    # ==================== CLINICAL DECISION ====================
    print_section("CLINICAL DECISION SUPPORT")
    
    print("RECOMMENDATION:")
    if ulcer_sim.amputation_risk_percent < 30:
        print("  ✅ PROCEED WITH MEDICAL MANAGEMENT")
        print("  • Amputation risk significantly reduced")
        print("  • Continue aggressive wound care")
        print("  • Re-evaluate in 4 weeks")
    else:
        print("  ⚠️  MEDICAL MANAGEMENT WITH CLOSE MONITORING")
        print("  • Amputation risk remains elevated")
        print("  • Continue treatment for 4 more weeks")
        print("  • Consider vascular surgery consultation")
        print("  • Prepare for possible amputation if no improvement")
    
    print("\nKEY SUCCESS FACTORS:")
    print("  1. Strict glycemic control (target HbA1c <7%)")
    print("  2. Aggressive infection management")
    print("  3. Vascular enhancement")
    print("  4. Optimal wound care with offloading")
    print("  5. Nutritional support")
    print("  6. Weekly monitoring and debridement as needed")
    
    # ==================== SUMMARY ====================
    print_section("SYSTEM SUMMARY")
    
    print(f"Total drugs recommended: {len(protocol.drugs)}")
    print(f"Categories addressed: {len(set(d.category for d in protocol.drugs))}")
    print(f"Safety warnings: {len(protocol.safety_warnings)}")
    print(f"Monitoring parameters: {len(protocol.monitoring_plan)}")
    print(f"\nInitial amputation risk: {create_wagner_grade_4_ulcer().amputation_risk_percent:.0f}%")
    print(f"Projected amputation risk (12 weeks): {ulcer_sim.amputation_risk_percent:.0f}%")
    print(f"Risk reduction: {create_wagner_grade_4_ulcer().amputation_risk_percent - ulcer_sim.amputation_risk_percent:.0f} percentage points")
    
    print("\n" + "="*80)
    print(" "*25 + "DEMONSTRATION COMPLETE")
    print("="*80 + "\n")


if __name__ == "__main__":
    main()
