"""
Prescribing Cascade Clean-Up Simulation.

Demonstrates the Deprescribing Engine identifying and removing 
harmful medication cascades in an 85-year-old patient.
"""

from body_state_model import HumanBodyState
from organs.geriatrics import GeriatricSpecialist
from drugs.deprescribing_engine import DeprescribingEngine

def run_deprescribing_demo():
    print("="*80)
    print("DEMO: INVERSE DEPRESCRIBING OPTIMIZER")
    print("="*80)
    
    # 1. Initialize Frail Patient
    patient = HumanBodyState()
    age = 85.0
    print(f"\n[PATIENT] Initializing 85yo Virtual Twin...")
    GeriatricSpecialist.apply_senescence(patient, age)
    
    # 2. Setup "The Dirty Dozen" Regimen (Prescribing Cascade)
    # Amlodipine (for BP) -> causes Edema -> Furosemide (for Edema) -> causes Electrolyte shift
    # Omeprazole (for GERD) -> long-term use causing malabsorption
    # Alprazolam (for Sleep/Anxiety) -> causing cognitive slowing
    # Lisinopril & Metformin (standard chronic care)
    active_regimen = [
        {"name": "Lisinopril", "dose": 20.0},
        {"name": "Metformin", "dose": 1000.0},
        {"name": "Amlodipine", "dose": 10.0},     # The Trigger
        {"name": "Furosemide", "dose": 40.0},      # The Cascade Add-on
        {"name": "Omeprazole", "dose": 40.0},      # Inappropriate long-term
        {"name": "Alprazolam", "dose": 0.5},       # Beers PIM
    ]
    
    # 3. Analyze Regimen
    engine = DeprescribingEngine()
    analysis = engine.analyze_polypharmacy(patient, active_regimen)
    
    # 4. Global Clean-up
    clean_results = engine.global_cleanup(patient, active_regimen)
    
    # 5. Output
    print("\n" + "="*80)
    print("DEPRESCRIBING INTELLIGENCE REPORT")
    print("="*80)
    print(f"Step 1: Identified {len(clean_results['removed'])} medications as potentially harmful.")
    print(f"Step 2: Predicted Homeostatic Rebound: +{clean_results['rebound']:.2f}%")
    print(f"Step 3: Final Optimized Balance: {clean_results['final']:.1f}%")
    
    print("\n[RECOMMENDATION] Discontinue the following to restore physiological reserve:")
    for drug in clean_results["removed"]:
        print(f" - {drug}")

    print("\n[VERDICT] Successful prescribing cascade resolution: Amlodipine & Furosemide eliminated.")
    print("[VERDICT] CNS/Metabolic optimization achieved via Alprazolam & Omeprazole removal.")

if __name__ == "__main__":
    run_deprescribing_demo()
