"""
Specialized Drug Library for Diabetic Foot Ulcer Treatment.

Comprehensive collection of drugs targeting all pathophysiological mechanisms:
- Wound healing agents
- Antibiotics (broad spectrum coverage)
- Glycemic control agents
- Vascular enhancement drugs
- Anti-inflammatory agents
- Pain management (neuropathic)
"""

from dataclasses import dataclass, field
from typing import Dict, List, Any, Optional
from enum import Enum


class DrugCategory(Enum):
    """Drug categories for ulcer treatment."""
    WOUND_HEALING = "wound_healing"
    ANTIBIOTIC = "antibiotic"
    GLYCEMIC_CONTROL = "glycemic_control"
    VASCULAR = "vascular"
    ANTI_INFLAMMATORY = "anti_inflammatory"
    PAIN_MANAGEMENT = "pain_management"
    NUTRITIONAL = "nutritional"


class RouteOfAdministration(Enum):
    """Routes of drug administration."""
    TOPICAL = "topical"
    ORAL = "oral"
    IV = "intravenous"
    IM = "intramuscular"
    SC = "subcutaneous"


@dataclass
class UlcerDrug:
    """Drug for diabetic ulcer treatment."""
    
    name: str
    category: DrugCategory
    mechanism: str
    route: RouteOfAdministration
    
    # Dosing
    standard_dose: str
    frequency: str
    duration_days: int
    
    # Effects on ulcer state (parameter: change per day)
    ulcer_effects: Dict[str, float] = field(default_factory=dict)
    
    # Safety
    contraindications: List[str] = field(default_factory=list)
    drug_interactions: List[str] = field(default_factory=list)
    renal_adjustment_required: bool = False
    hepatic_adjustment_required: bool = False
    
    # Monitoring
    monitoring_parameters: List[str] = field(default_factory=list)
    
    # Cost and availability
    cost_category: str = "moderate"  # low, moderate, high
    
    def get_daily_effects(self) -> Dict[str, float]:
        """Get daily effects on ulcer parameters."""
        return self.ulcer_effects.copy()


# ==================== WOUND HEALING AGENTS ====================

BECAPLERMIN = UlcerDrug(
    name="Becaplermin (Regranex)",
    category=DrugCategory.WOUND_HEALING,
    mechanism="Recombinant PDGF-BB, promotes granulation tissue formation",
    route=RouteOfAdministration.TOPICAL,
    standard_dose="Apply 2% gel once daily",
    frequency="Once daily",
    duration_days=56,
    ulcer_effects={
        "granulation_tissue_percent": 2.0,  # +2% per day
        "epithelialization_percent": 1.0,
        "healing_rate_percent_per_week": 5.0
    },
    contraindications=["malignancy at application site", "necrotic tissue >50%"],
    monitoring_parameters=["wound size", "granulation tissue"],
    cost_category="high"
)

COLLAGEN_DRESSING = UlcerDrug(
    name="Collagen Wound Dressing",
    category=DrugCategory.WOUND_HEALING,
    mechanism="Provides scaffold for cell migration, absorbs exudate",
    route=RouteOfAdministration.TOPICAL,
    standard_dose="Apply to wound bed, change every 3-7 days",
    frequency="Every 3-7 days",
    duration_days=84,
    ulcer_effects={
        "granulation_tissue_percent": 1.5,
        "exudate_amount": -0.5,  # Reduces exudate
        "healing_rate_percent_per_week": 3.0
    },
    contraindications=["sensitivity to bovine/porcine products"],
    monitoring_parameters=["wound bed appearance", "exudate amount"],
    cost_category="moderate"
)

SILVER_SULFADIAZINE = UlcerDrug(
    name="Silver Sulfadiazine 1% Cream",
    category=DrugCategory.WOUND_HEALING,
    mechanism="Antimicrobial + promotes epithelialization",
    route=RouteOfAdministration.TOPICAL,
    standard_dose="Apply 1/16 inch thickness twice daily",
    frequency="Twice daily",
    duration_days=14,
    ulcer_effects={
        "bacterial_load_cfu_g": -1e5,  # Reduces bacterial load
        "epithelialization_percent": 0.8,
        "necrotic_tissue_percent": -1.0
    },
    contraindications=["sulfa allergy", "G6PD deficiency", "pregnancy"],
    monitoring_parameters=["wound appearance", "signs of infection"],
    cost_category="low"
)

# ==================== ANTIBIOTICS ====================

VANCOMYCIN = UlcerDrug(
    name="Vancomycin",
    category=DrugCategory.ANTIBIOTIC,
    mechanism="Glycopeptide antibiotic, MRSA coverage",
    route=RouteOfAdministration.IV,
    standard_dose="15-20 mg/kg IV every 8-12 hours",
    frequency="Every 8-12 hours",
    duration_days=14,
    ulcer_effects={
        "bacterial_load_cfu_g": -5e6,
        "cellulitis_diameter_cm": -0.5,
        "white_blood_cell_count_k_ul": -0.5,
        "c_reactive_protein_mg_l": -5.0
    },
    contraindications=["vancomycin allergy"],
    drug_interactions=["aminoglycosides (nephrotoxicity)"],
    renal_adjustment_required=True,
    monitoring_parameters=["trough levels", "renal function", "CBC"],
    cost_category="high"
)

PIPERACILLIN_TAZOBACTAM = UlcerDrug(
    name="Piperacillin-Tazobactam (Zosyn)",
    category=DrugCategory.ANTIBIOTIC,
    mechanism="Beta-lactam + beta-lactamase inhibitor, broad spectrum",
    route=RouteOfAdministration.IV,
    standard_dose="3.375-4.5g IV every 6-8 hours",
    frequency="Every 6-8 hours",
    duration_days=14,
    ulcer_effects={
        "bacterial_load_cfu_g": -6e6,
        "cellulitis_diameter_cm": -0.6,
        "purulent_drainage": -0.3,
        "c_reactive_protein_mg_l": -6.0
    },
    contraindications=["penicillin allergy"],
    renal_adjustment_required=True,
    monitoring_parameters=["renal function", "CBC", "LFTs"],
    cost_category="moderate"
)

LINEZOLID = UlcerDrug(
    name="Linezolid (Zyvox)",
    category=DrugCategory.ANTIBIOTIC,
    mechanism="Oxazolidinone, excellent tissue penetration, MRSA coverage",
    route=RouteOfAdministration.ORAL,
    standard_dose="600 mg PO twice daily",
    frequency="Twice daily",
    duration_days=14,
    ulcer_effects={
        "bacterial_load_cfu_g": -4e6,
        "cellulitis_diameter_cm": -0.4,
        "biofilm_present": -0.2
    },
    contraindications=["MAOI use", "uncontrolled hypertension"],
    drug_interactions=["SSRIs (serotonin syndrome)", "MAOIs"],
    monitoring_parameters=["CBC (thrombocytopenia)", "visual changes"],
    cost_category="high"
)

METRONIDAZOLE = UlcerDrug(
    name="Metronidazole",
    category=DrugCategory.ANTIBIOTIC,
    mechanism="Nitroimidazole, anaerobic coverage",
    route=RouteOfAdministration.IV,
    standard_dose="500 mg IV every 8 hours",
    frequency="Every 8 hours",
    duration_days=14,
    ulcer_effects={
        "bacterial_load_cfu_g": -3e6,
        "necrotic_tissue_percent": -1.5
    },
    contraindications=["first trimester pregnancy", "alcohol use"],
    drug_interactions=["warfarin (increased INR)", "alcohol (disulfiram reaction)"],
    monitoring_parameters=["neurological symptoms"],
    cost_category="low"
)

# ==================== GLYCEMIC CONTROL ====================

INSULIN_REGULAR = UlcerDrug(
    name="Regular Insulin (IV)",
    category=DrugCategory.GLYCEMIC_CONTROL,
    mechanism="Tight glycemic control, promotes wound healing",
    route=RouteOfAdministration.IV,
    standard_dose="Continuous infusion, titrate to glucose 140-180 mg/dL",
    frequency="Continuous",
    duration_days=14,
    ulcer_effects={
        "blood_glucose_mg_dl": -50.0,  # Target reduction
        "hba1c_percent": -0.3,  # Over weeks
        "healing_rate_percent_per_week": 2.0
    },
    contraindications=["hypoglycemia"],
    monitoring_parameters=["blood glucose hourly", "potassium"],
    cost_category="low"
)

METFORMIN = UlcerDrug(
    name="Metformin",
    category=DrugCategory.GLYCEMIC_CONTROL,
    mechanism="Biguanide, improves insulin sensitivity",
    route=RouteOfAdministration.ORAL,
    standard_dose="500-1000 mg twice daily",
    frequency="Twice daily",
    duration_days=90,
    ulcer_effects={
        "blood_glucose_mg_dl": -30.0,
        "hba1c_percent": -1.0,
        "healing_rate_percent_per_week": 1.5
    },
    contraindications=["GFR <30", "lactic acidosis risk", "contrast dye"],
    renal_adjustment_required=True,
    monitoring_parameters=["renal function", "vitamin B12"],
    cost_category="low"
)

# ==================== VASCULAR ENHANCEMENT ====================

CILOSTAZOL = UlcerDrug(
    name="Cilostazol (Pletal)",
    category=DrugCategory.VASCULAR,
    mechanism="Phosphodiesterase-3 inhibitor, antiplatelet + vasodilation",
    route=RouteOfAdministration.ORAL,
    standard_dose="100 mg twice daily",
    frequency="Twice daily",
    duration_days=90,
    ulcer_effects={
        "tissue_perfusion_percent": 5.0,
        "ankle_brachial_index": 0.05,
        "transcutaneous_oxygen_mmhg": 3.0,
        "healing_rate_percent_per_week": 2.0
    },
    contraindications=["heart failure", "bleeding disorder"],
    drug_interactions=["CYP3A4/2C19 inhibitors"],
    monitoring_parameters=["bleeding", "heart rate"],
    cost_category="moderate"
)

PENTOXIFYLLINE = UlcerDrug(
    name="Pentoxifylline (Trental)",
    category=DrugCategory.VASCULAR,
    mechanism="Improves RBC deformability, microcirculation",
    route=RouteOfAdministration.ORAL,
    standard_dose="400 mg three times daily",
    frequency="Three times daily",
    duration_days=90,
    ulcer_effects={
        "tissue_perfusion_percent": 3.0,
        "transcutaneous_oxygen_mmhg": 2.0,
        "healing_rate_percent_per_week": 1.5
    },
    contraindications=["recent cerebral/retinal hemorrhage"],
    drug_interactions=["warfarin (increased bleeding)"],
    monitoring_parameters=["bleeding", "GI upset"],
    cost_category="low"
)

ASPIRIN = UlcerDrug(
    name="Aspirin",
    category=DrugCategory.VASCULAR,
    mechanism="Antiplatelet, improves microvascular flow",
    route=RouteOfAdministration.ORAL,
    standard_dose="81-325 mg once daily",
    frequency="Once daily",
    duration_days=365,
    ulcer_effects={
        "tissue_perfusion_percent": 2.0,
        "healing_rate_percent_per_week": 1.0
    },
    contraindications=["active bleeding", "aspirin allergy"],
    drug_interactions=["warfarin", "NSAIDs"],
    monitoring_parameters=["bleeding", "GI symptoms"],
    cost_category="low"
)

# ==================== PAIN MANAGEMENT ====================

GABAPENTIN = UlcerDrug(
    name="Gabapentin (Neurontin)",
    category=DrugCategory.PAIN_MANAGEMENT,
    mechanism="Calcium channel modulator, neuropathic pain",
    route=RouteOfAdministration.ORAL,
    standard_dose="300-1200 mg three times daily",
    frequency="Three times daily",
    duration_days=90,
    ulcer_effects={
        "pain_score_0_10": -3.0,
        "neuropathic_pain_present": -0.5
    },
    contraindications=["gabapentin allergy"],
    renal_adjustment_required=True,
    monitoring_parameters=["sedation", "dizziness", "renal function"],
    cost_category="low"
)

PREGABALIN = UlcerDrug(
    name="Pregabalin (Lyrica)",
    category=DrugCategory.PAIN_MANAGEMENT,
    mechanism="Alpha-2-delta ligand, neuropathic pain",
    route=RouteOfAdministration.ORAL,
    standard_dose="75-150 mg twice daily",
    frequency="Twice daily",
    duration_days=90,
    ulcer_effects={
        "pain_score_0_10": -3.5,
        "neuropathic_pain_present": -0.6
    },
    contraindications=["pregabalin allergy"],
    renal_adjustment_required=True,
    monitoring_parameters=["sedation", "weight gain", "edema"],
    cost_category="moderate"
)

DULOXETINE = UlcerDrug(
    name="Duloxetine (Cymbalta)",
    category=DrugCategory.PAIN_MANAGEMENT,
    mechanism="SNRI, neuropathic pain",
    route=RouteOfAdministration.ORAL,
    standard_dose="60 mg once daily",
    frequency="Once daily",
    duration_days=90,
    ulcer_effects={
        "pain_score_0_10": -2.5,
        "neuropathic_pain_present": -0.4
    },
    contraindications=["MAOI use", "uncontrolled glaucoma"],
    drug_interactions=["MAOIs", "CYP1A2 inhibitors"],
    hepatic_adjustment_required=True,
    monitoring_parameters=["blood pressure", "liver function"],
    cost_category="moderate"
)

# ==================== NUTRITIONAL SUPPORT ====================

PROTEIN_SUPPLEMENT = UlcerDrug(
    name="High-Protein Nutritional Supplement",
    category=DrugCategory.NUTRITIONAL,
    mechanism="Provides amino acids for collagen synthesis",
    route=RouteOfAdministration.ORAL,
    standard_dose="1.2-1.5 g/kg/day protein",
    frequency="With meals",
    duration_days=90,
    ulcer_effects={
        "albumin_g_dl": 0.1,
        "healing_rate_percent_per_week": 1.0,
        "granulation_tissue_percent": 0.5
    },
    contraindications=["severe renal failure"],
    monitoring_parameters=["albumin", "prealbumin"],
    cost_category="low"
)

VITAMIN_C = UlcerDrug(
    name="Vitamin C (Ascorbic Acid)",
    category=DrugCategory.NUTRITIONAL,
    mechanism="Cofactor for collagen synthesis",
    route=RouteOfAdministration.ORAL,
    standard_dose="500-1000 mg daily",
    frequency="Once daily",
    duration_days=90,
    ulcer_effects={
        "granulation_tissue_percent": 0.5,
        "healing_rate_percent_per_week": 0.5
    },
    contraindications=["oxalate kidney stones"],
    monitoring_parameters=["none"],
    cost_category="low"
)

ZINC_SULFATE = UlcerDrug(
    name="Zinc Sulfate",
    category=DrugCategory.NUTRITIONAL,
    mechanism="Cofactor for metalloenzymes in wound healing",
    route=RouteOfAdministration.ORAL,
    standard_dose="220 mg daily (50 mg elemental zinc)",
    frequency="Once daily",
    duration_days=90,
    ulcer_effects={
        "epithelialization_percent": 0.5,
        "healing_rate_percent_per_week": 0.5
    },
    contraindications=["zinc toxicity"],
    monitoring_parameters=["copper levels (if long-term)"],
    cost_category="low"
)


# ==================== DRUG LIBRARY ====================

ULCER_DRUG_LIBRARY = {
    # Wound healing
    "becaplermin": BECAPLERMIN,
    "collagen_dressing": COLLAGEN_DRESSING,
    "silver_sulfadiazine": SILVER_SULFADIAZINE,
    
    # Antibiotics
    "vancomycin": VANCOMYCIN,
    "piperacillin_tazobactam": PIPERACILLIN_TAZOBACTAM,
    "linezolid": LINEZOLID,
    "metronidazole": METRONIDAZOLE,
    
    # Glycemic control
    "insulin_regular": INSULIN_REGULAR,
    "metformin": METFORMIN,
    
    # Vascular
    "cilostazol": CILOSTAZOL,
    "pentoxifylline": PENTOXIFYLLINE,
    "aspirin": ASPIRIN,
    
    # Pain management
    "gabapentin": GABAPENTIN,
    "pregabalin": PREGABALIN,
    "duloxetine": DULOXETINE,
    
    # Nutritional
    "protein_supplement": PROTEIN_SUPPLEMENT,
    "vitamin_c": VITAMIN_C,
    "zinc_sulfate": ZINC_SULFATE
}


def get_drugs_by_category(category: DrugCategory) -> List[UlcerDrug]:
    """Get all drugs in a specific category."""
    return [drug for drug in ULCER_DRUG_LIBRARY.values() if drug.category == category]


def get_drug(drug_name: str) -> Optional[UlcerDrug]:
    """Get drug by name."""
    return ULCER_DRUG_LIBRARY.get(drug_name.lower())


def list_all_drugs() -> List[str]:
    """List all available drugs."""
    return list(ULCER_DRUG_LIBRARY.keys())


def check_drug_interactions(drug_list: List[str]) -> List[str]:
    """Check for drug interactions in a combination."""
    interactions = []
    drugs = [get_drug(name) for name in drug_list if get_drug(name)]
    
    for i, drug1 in enumerate(drugs):
        for drug2 in drugs[i+1:]:
            if drug2.name in drug1.drug_interactions:
                interactions.append(f"{drug1.name} + {drug2.name}: Interaction detected")
            if drug1.name in drug2.drug_interactions:
                interactions.append(f"{drug2.name} + {drug1.name}: Interaction detected")
    
    return interactions


if __name__ == "__main__":
    print("=== Diabetic Ulcer Drug Library ===\n")
    print(f"Total drugs: {len(ULCER_DRUG_LIBRARY)}\n")
    
    for category in DrugCategory:
        drugs = get_drugs_by_category(category)
        print(f"\n{category.value.upper().replace('_', ' ')} ({len(drugs)} drugs):")
        for drug in drugs:
            print(f"  - {drug.name}")
            print(f"    Mechanism: {drug.mechanism}")
            print(f"    Route: {drug.route.value}")
            print(f"    Dose: {drug.standard_dose}")
