"""
Diabetic Ulcer Drug Discovery Engine.

Multi-objective optimization engine that:
1. Assesses ulcer state and identifies deficits
2. Selects drugs targeting each pathophysiological mechanism
3. Optimizes combinations for synergy and safety
4. Generates treatment protocols with dosing
"""

import sys
from pathlib import Path
sys.path.append(str(Path(__file__).parent.parent))

from typing import Dict, List, Any, Tuple, Optional
from dataclasses import dataclass, field
from diseases.diabetic_ulcer_state import DiabeticUlcerState, WagnerGrade, InfectionSeverity
from drugs.ulcer_drug_library import (
    UlcerDrug, DrugCategory, ULCER_DRUG_LIBRARY,
    get_drugs_by_category, check_drug_interactions
)


@dataclass
class TreatmentProtocol:
    """Complete treatment protocol for diabetic ulcer."""
    
    drugs: List[UlcerDrug] = field(default_factory=list)
    rationale: Dict[str, str] = field(default_factory=dict)
    safety_warnings: List[str] = field(default_factory=list)
    monitoring_plan: List[str] = field(default_factory=list)
    expected_outcomes: Dict[str, Any] = field(default_factory=dict)
    
    def get_summary(self) -> str:
        """Get human-readable treatment protocol."""
        summary = "=== TREATMENT PROTOCOL ===\n\n"
        
        summary += f"Total Medications: {len(self.drugs)}\n\n"
        
        # Group by category
        by_category = {}
        for drug in self.drugs:
            cat = drug.category.value
            if cat not in by_category:
                by_category[cat] = []
            by_category[cat].append(drug)
        
        for category, drugs in by_category.items():
            summary += f"\n{category.upper().replace('_', ' ')}:\n"
            for drug in drugs:
                summary += f"  • {drug.name}\n"
                summary += f"    Dose: {drug.standard_dose}\n"
                summary += f"    Frequency: {drug.frequency}\n"
                summary += f"    Duration: {drug.duration_days} days\n"
                summary += f"    Route: {drug.route.value}\n"
                if category in self.rationale:
                    summary += f"    Rationale: {self.rationale[category]}\n"
        
        if self.safety_warnings:
            summary += "\n⚠️  SAFETY WARNINGS:\n"
            for warning in self.safety_warnings:
                summary += f"  • {warning}\n"
        
        if self.monitoring_plan:
            summary += "\nMONITORING PLAN:\n"
            for item in self.monitoring_plan:
                summary += f"  • {item}\n"
        
        if self.expected_outcomes:
            summary += "\nEXPECTED OUTCOMES:\n"
            for outcome, value in self.expected_outcomes.items():
                summary += f"  • {outcome}: {value}\n"
        
        return summary


class UlcerDrugDiscoveryEngine:
    """
    Drug discovery engine for diabetic foot ulcers.
    
    Uses multi-objective optimization to find optimal drug combinations
    addressing all pathophysiological deficits.
    """
    
    def __init__(self):
        """Initialize discovery engine."""
        self.drug_library = ULCER_DRUG_LIBRARY
    
    def discover_treatment(
        self,
        ulcer_state: DiabeticUlcerState,
        patient_contraindications: Optional[List[str]] = None,
        renal_impairment: bool = False,
        hepatic_impairment: bool = False
    ) -> TreatmentProtocol:
        """
        Discover optimal drug combination for ulcer treatment.
        
        Args:
            ulcer_state: Current ulcer state
            patient_contraindications: List of patient-specific contraindications
            renal_impairment: Whether patient has renal impairment
            hepatic_impairment: Whether patient has hepatic impairment
        
        Returns:
            Complete treatment protocol
        """
        protocol = TreatmentProtocol()
        patient_contraindications = patient_contraindications or []
        
        # Step 1: Identify critical deficits
        deficits = ulcer_state.get_critical_deficits()
        
        # Step 2: Select drugs for each deficit
        selected_drugs = []
        
        # INFECTION - highest priority
        if "infection" in deficits:
            infection_drugs = self._select_antibiotics(
                deficits["infection"],
                patient_contraindications,
                renal_impairment
            )
            selected_drugs.extend(infection_drugs)
            protocol.rationale["antibiotic"] = (
                f"Treating {deficits['infection']['severity']} infection with "
                f"bacterial load {deficits['infection']['bacterial_load']:.1e} CFU/g"
            )
        
        # ISCHEMIA - critical for healing
        if "ischemia" in deficits:
            vascular_drugs = self._select_vascular_agents(
                deficits["ischemia"],
                patient_contraindications
            )
            selected_drugs.extend(vascular_drugs)
            protocol.rationale["vascular"] = (
                f"Improving perfusion (current: {deficits['ischemia']['perfusion_percent']:.0f}%, "
                f"ABI: {deficits['ischemia']['abi']:.2f})"
            )
        
        # HYPERGLYCEMIA - essential for healing
        if "hyperglycemia" in deficits:
            glycemic_drugs = self._select_glycemic_agents(
                deficits["hyperglycemia"],
                patient_contraindications,
                renal_impairment,
                ulcer_state.wagner_grade
            )
            selected_drugs.extend(glycemic_drugs)
            protocol.rationale["glycemic_control"] = (
                f"Optimizing glucose control (HbA1c: {deficits['hyperglycemia']['hba1c']:.1f}%)"
            )
        
        # WOUND BED - promote healing
        if "wound_bed" in deficits or ulcer_state.wagner_grade.value >= 2:
            wound_drugs = self._select_wound_healing_agents(
                ulcer_state,
                patient_contraindications
            )
            selected_drugs.extend(wound_drugs)
            protocol.rationale["wound_healing"] = (
                f"Promoting granulation tissue formation (current: "
                f"{ulcer_state.granulation_tissue_percent:.0f}%)"
            )
        
        # PAIN - quality of life
        if "pain" in deficits:
            pain_drugs = self._select_pain_management(
                deficits["pain"],
                patient_contraindications,
                renal_impairment,
                hepatic_impairment
            )
            selected_drugs.extend(pain_drugs)
            protocol.rationale["pain_management"] = (
                f"Managing {'neuropathic' if deficits['pain']['neuropathic'] else 'nociceptive'} "
                f"pain (score: {deficits['pain']['score']}/10)"
            )
        
        # MALNUTRITION - support healing
        if "malnutrition" in deficits:
            nutritional_drugs = self._select_nutritional_support(
                deficits["malnutrition"]
            )
            selected_drugs.extend(nutritional_drugs)
            protocol.rationale["nutritional"] = (
                f"Supporting wound healing (albumin: {deficits['malnutrition']['albumin']:.1f} g/dL)"
            )
        
        protocol.drugs = selected_drugs
        
        # Step 3: Safety validation
        self._validate_safety(protocol, patient_contraindications, renal_impairment, hepatic_impairment)
        
        # Step 4: Generate monitoring plan
        self._generate_monitoring_plan(protocol)
        
        # Step 5: Predict outcomes
        self._predict_outcomes(protocol, ulcer_state)
        
        return protocol
    
    def _select_antibiotics(
        self,
        infection_details: Dict[str, Any],
        contraindications: List[str],
        renal_impairment: bool
    ) -> List[UlcerDrug]:
        """Select appropriate antibiotics based on infection severity."""
        severity = infection_details["severity"]
        systemic = infection_details.get("systemic", False)
        
        selected = []
        
        if severity == "severe" or systemic:
            # Severe infection: broad spectrum IV coverage
            # MRSA coverage
            if "vancomycin allergy" not in contraindications:
                selected.append(self.drug_library["vancomycin"])
            else:
                selected.append(self.drug_library["linezolid"])
            
            # Gram-negative + anaerobic coverage
            if "penicillin allergy" not in contraindications:
                selected.append(self.drug_library["piperacillin_tazobactam"])
            
            # Anaerobic coverage
            selected.append(self.drug_library["metronidazole"])
        
        elif severity == "moderate":
            # Moderate infection: broad coverage
            if "penicillin allergy" not in contraindications:
                selected.append(self.drug_library["piperacillin_tazobactam"])
            else:
                selected.append(self.drug_library["linezolid"])
            
            selected.append(self.drug_library["metronidazole"])
        
        else:  # mild
            # Mild infection: topical + oral
            if "sulfa allergy" not in contraindications:
                selected.append(self.drug_library["silver_sulfadiazine"])
            selected.append(self.drug_library["linezolid"])
        
        return selected
    
    def _select_vascular_agents(
        self,
        ischemia_details: Dict[str, Any],
        contraindications: List[str]
    ) -> List[UlcerDrug]:
        """Select vascular enhancement drugs."""
        selected = []
        
        # Cilostazol - most effective but contraindicated in heart failure
        if "heart failure" not in contraindications:
            selected.append(self.drug_library["cilostazol"])
        
        # Pentoxifylline - safe alternative
        selected.append(self.drug_library["pentoxifylline"])
        
        # Aspirin - unless contraindicated
        if "active bleeding" not in contraindications and "aspirin allergy" not in contraindications:
            selected.append(self.drug_library["aspirin"])
        
        return selected
    
    def _select_glycemic_agents(
        self,
        hyperglycemia_details: Dict[str, Any],
        contraindications: List[str],
        renal_impairment: bool,
        wagner_grade: WagnerGrade
    ) -> List[UlcerDrug]:
        """Select glycemic control agents."""
        selected = []
        hba1c = hyperglycemia_details["hba1c"]
        
        # Severe ulcer (Wagner 3-5) or very high HbA1c: IV insulin
        if wagner_grade.value >= 3 or hba1c > 9.0:
            selected.append(self.drug_library["insulin_regular"])
        
        # Add metformin if no renal impairment
        if not renal_impairment and hba1c > 7.5:
            selected.append(self.drug_library["metformin"])
        
        return selected
    
    def _select_wound_healing_agents(
        self,
        ulcer_state: DiabeticUlcerState,
        contraindications: List[str]
    ) -> List[UlcerDrug]:
        """Select wound healing agents."""
        selected = []
        
        # Becaplermin for Wagner 1-3 with good perfusion
        if (ulcer_state.wagner_grade.value <= 3 and 
            ulcer_state.tissue_perfusion_percent > 40 and
            ulcer_state.necrotic_tissue_percent < 50):
            selected.append(self.drug_library["becaplermin"])
        
        # Collagen dressing for all grades
        selected.append(self.drug_library["collagen_dressing"])
        
        return selected
    
    def _select_pain_management(
        self,
        pain_details: Dict[str, Any],
        contraindications: List[str],
        renal_impairment: bool,
        hepatic_impairment: bool
    ) -> List[UlcerDrug]:
        """Select pain management drugs."""
        selected = []
        neuropathic = pain_details["neuropathic"]
        
        if neuropathic:
            # Pregabalin - most effective for neuropathic pain
            if not renal_impairment:
                selected.append(self.drug_library["pregabalin"])
            else:
                # Gabapentin with renal adjustment
                selected.append(self.drug_library["gabapentin"])
        else:
            # Duloxetine for non-neuropathic
            if not hepatic_impairment:
                selected.append(self.drug_library["duloxetine"])
        
        return selected
    
    def _select_nutritional_support(
        self,
        malnutrition_details: Dict[str, Any]
    ) -> List[UlcerDrug]:
        """Select nutritional supplements."""
        selected = [
            self.drug_library["protein_supplement"],
            self.drug_library["vitamin_c"],
            self.drug_library["zinc_sulfate"]
        ]
        return selected
    
    def _validate_safety(
        self,
        protocol: TreatmentProtocol,
        contraindications: List[str],
        renal_impairment: bool,
        hepatic_impairment: bool
    ):
        """Validate safety of drug combination."""
        # Check drug interactions
        drug_names = [drug.name for drug in protocol.drugs]
        interactions = check_drug_interactions([d.lower().replace(" ", "_").replace("(", "").replace(")", "").split()[0] 
                                                for d in drug_names])
        protocol.safety_warnings.extend(interactions)
        
        # Check contraindications
        for drug in protocol.drugs:
            for contra in drug.contraindications:
                if contra in contraindications:
                    protocol.safety_warnings.append(
                        f"⚠️  {drug.name} contraindicated: {contra}"
                    )
        
        # Check renal/hepatic adjustments
        if renal_impairment:
            for drug in protocol.drugs:
                if drug.renal_adjustment_required:
                    protocol.safety_warnings.append(
                        f"⚠️  {drug.name} requires renal dose adjustment"
                    )
        
        if hepatic_impairment:
            for drug in protocol.drugs:
                if drug.hepatic_adjustment_required:
                    protocol.safety_warnings.append(
                        f"⚠️  {drug.name} requires hepatic dose adjustment"
                    )
    
    def _generate_monitoring_plan(self, protocol: TreatmentProtocol):
        """Generate comprehensive monitoring plan."""
        monitoring_set = set()
        
        for drug in protocol.drugs:
            monitoring_set.update(drug.monitoring_parameters)
        
        # Add standard ulcer monitoring
        monitoring_set.add("Wound size measurement weekly")
        monitoring_set.add("Wound bed assessment (granulation, necrosis)")
        monitoring_set.add("Signs of infection (erythema, drainage, odor)")
        monitoring_set.add("Perfusion assessment (pulses, capillary refill)")
        monitoring_set.add("Blood glucose monitoring")
        
        protocol.monitoring_plan = sorted(list(monitoring_set))
    
    def _predict_outcomes(self, protocol: TreatmentProtocol, ulcer_state: DiabeticUlcerState):
        """Predict treatment outcomes."""
        # Calculate cumulative effects
        total_healing_rate = 0.0
        total_bacterial_reduction = 0.0
        total_perfusion_improvement = 0.0
        
        for drug in protocol.drugs:
            effects = drug.get_daily_effects()
            total_healing_rate += effects.get("healing_rate_percent_per_week", 0.0)
            total_bacterial_reduction += abs(effects.get("bacterial_load_cfu_g", 0.0))
            total_perfusion_improvement += effects.get("tissue_perfusion_percent", 0.0)
        
        # Estimate healing time
        baseline_weeks = ulcer_state.weeks_to_heal_estimate
        improvement_factor = 1.0 + (total_healing_rate / 100.0)
        estimated_weeks = baseline_weeks / improvement_factor if improvement_factor > 1.0 else baseline_weeks
        
        protocol.expected_outcomes = {
            "Estimated healing time": f"{estimated_weeks:.0f} weeks (baseline: {baseline_weeks:.0f} weeks)",
            "Expected healing rate": f"{total_healing_rate:.1f}% per week",
            "Amputation risk reduction": f"{ulcer_state.amputation_risk_percent:.0f}% → {max(5, ulcer_state.amputation_risk_percent - 30):.0f}%",
            "Infection control": "Expected within 7-14 days" if total_bacterial_reduction > 0 else "N/A",
            "Perfusion improvement": f"+{total_perfusion_improvement:.0f}%" if total_perfusion_improvement > 0 else "N/A"
        }


# ==================== EXAMPLE USAGE ====================

if __name__ == "__main__":
    from diseases.diabetic_ulcer_state import create_wagner_grade_4_ulcer
    
    print("="*70)
    print("DIABETIC ULCER DRUG DISCOVERY ENGINE - DEMONSTRATION")
    print("="*70)
    
    # Create critical ulcer case
    print("\n1. Creating Wagner Grade 4 ulcer (amputation stage)...\n")
    ulcer = create_wagner_grade_4_ulcer()
    print(ulcer.get_state_summary())
    
    # Initialize discovery engine
    print("\n" + "="*70)
    print("2. Running drug discovery engine...\n")
    engine = UlcerDrugDiscoveryEngine()
    
    # Discover treatment
    protocol = engine.discover_treatment(
        ulcer_state=ulcer,
        patient_contraindications=["heart failure"],  # Example contraindication
        renal_impairment=False,
        hepatic_impairment=False
    )
    
    # Display protocol
    print("\n" + "="*70)
    print("3. GENERATED TREATMENT PROTOCOL")
    print("="*70)
    print(protocol.get_summary())
    
    print("\n" + "="*70)
    print("DISCOVERY COMPLETE")
    print("="*70)
