"""
Comprehensive FDA-Approved Drug Library.

Organized by therapeutic class for multimorbidity management.
Includes high-priority drugs with PK/PD and safety envelopes.
"""

from dataclasses import dataclass
from typing import Dict, List
from .drug_intervention import DrugIntervention, RouteOfAdministration


# ==================== CARDIOVASCULAR DRUGS ====================

def create_amlodipine(dose_mg: float = 5.0) -> DrugIntervention:
    """Calcium channel blocker for hypertension."""
    return DrugIntervention(
        drug_name="Amlodipine",
        drug_class="Calcium Channel Blocker",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=64.0,
        absorption_rate_constant_per_hr=0.5,
        distribution_volume_l=21.0,
        clearance_l_hr=0.5,
        half_life_hours=35.0,
        protein_binding_percent=97.5,
        target_organs=["Heart"],
        therapeutic_effects={
            "Heart": {
                "afterload_mmhg": -15.0,
                "coronary_blood_flow_ml_min": 30.0
            }
        },
        therapeutic_index=10.0,
        toxic_concentration_mg_l=20.0
    )


def create_atorvastatin(dose_mg: float = 20.0) -> DrugIntervention:
    """Statin for cholesterol management."""
    return DrugIntervention(
        drug_name="Atorvastatin",
        drug_class="Statin",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=14.0,
        absorption_rate_constant_per_hr=1.0,
        distribution_volume_l=381.0,
        clearance_l_hr=30.0,
        half_life_hours=14.0,
        protein_binding_percent=98.0,
        target_organs=["Liver", "Heart"],
        therapeutic_effects={
            "Liver": {
                "cholesterol_synthesis_percent": -30.0
            },
            "Heart": {
                "plaque_stability_percent": 20.0
            }
        },
        therapeutic_index=15.0,
        toxic_concentration_mg_l=100.0
    )


def create_furosemide(dose_mg: float = 40.0) -> DrugIntervention:
    """Loop diuretic for heart failure and edema."""
    return DrugIntervention(
        drug_name="Furosemide",
        drug_class="Loop Diuretic",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=50.0,
        absorption_rate_constant_per_hr=1.5,
        distribution_volume_l=7.7,
        clearance_l_hr=2.0,
        half_life_hours=2.0,
        protein_binding_percent=98.0,
        target_organs=["Kidneys", "Heart"],
        therapeutic_effects={
            "Kidneys": {
                "urine_output_ml_day": 1000.0,
                "sodium_excretion_meq_day": 200.0
            },
            "Heart": {
                "preload_mmhg": -5.0
            }
        },
        therapeutic_index=5.0,
        toxic_concentration_mg_l=50.0
    )


def create_warfarin(dose_mg: float = 5.0) -> DrugIntervention:
    """Anticoagulant for thrombosis prevention."""
    return DrugIntervention(
        drug_name="Warfarin",
        drug_class="Anticoagulant",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=100.0,
        absorption_rate_constant_per_hr=1.2,
        distribution_volume_l=8.0,
        clearance_l_hr=0.2,
        half_life_hours=40.0,
        protein_binding_percent=99.0,
        target_organs=["Liver", "Blood"],
        therapeutic_effects={
            "Liver": {
                "clotting_factor_synthesis_percent": -50.0
            }
        },
        therapeutic_index=3.0,
        toxic_concentration_mg_l=10.0
    )


def create_digoxin(dose_mg: float = 0.25) -> DrugIntervention:
    """Cardiac glycoside for heart failure and atrial fibrillation."""
    return DrugIntervention(
        drug_name="Digoxin",
        drug_class="Cardiac Glycoside",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=70.0,
        absorption_rate_constant_per_hr=0.8,
        distribution_volume_l=500.0,
        clearance_l_hr=10.0,
        half_life_hours=36.0,
        protein_binding_percent=25.0,
        target_organs=["Heart"],
        therapeutic_effects={
            "Heart": {
                "contractility_percent": 15.0,
                "heart_rate_bpm": -10.0
            }
        },
        therapeutic_index=2.0,
        toxic_concentration_mg_l=2.0
    )


# ==================== ENDOCRINE/METABOLIC DRUGS ====================

def create_insulin_rapid(dose_units: float = 10.0) -> DrugIntervention:
    """Rapid-acting insulin for diabetes."""
    return DrugIntervention(
        drug_name="Insulin (Rapid-Acting)",
        drug_class="Insulin",
        dose_mg=dose_units,
        route=RouteOfAdministration.SC,
        bioavailability_percent=100.0,
        absorption_rate_constant_per_hr=4.0,
        distribution_volume_l=12.0,
        clearance_l_hr=10.0,
        half_life_hours=0.25,
        protein_binding_percent=0.0,
        target_organs=["Pancreas", "Liver", "Muscles"],
        therapeutic_effects={
            "Liver": {
                "glucose_production_mg_dl_hr": -100.0
            },
            "Muscles": {
                "glucose_uptake_percent": 100.0
            }
        },
        therapeutic_index=5.0,
        toxic_concentration_mg_l=100.0
    )


def create_levothyroxine(dose_mcg: float = 100.0) -> DrugIntervention:
    """Thyroid hormone replacement."""
    return DrugIntervention(
        drug_name="Levothyroxine",
        drug_class="Thyroid Hormone",
        dose_mg=dose_mcg / 1000.0,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=70.0,
        absorption_rate_constant_per_hr=0.5,
        distribution_volume_l=10.0,
        clearance_l_hr=0.05,
        half_life_hours=168.0,
        protein_binding_percent=99.97,
        target_organs=["Endocrine System", "Heart", "Muscles"],
        therapeutic_effects={
            "Endocrine System": {
                "t4_ug_dl": 4.0,
                "metabolic_rate_percent": 30.0
            },
            "Heart": {
                "heart_rate_bpm": 10.0
            }
        },
        therapeutic_index=10.0,
        toxic_concentration_mg_l=0.5
    )


def create_omeprazole(dose_mg: float = 20.0) -> DrugIntervention:
    """Proton pump inhibitor for acid reflux."""
    return DrugIntervention(
        drug_name="Omeprazole",
        drug_class="Proton Pump Inhibitor",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=40.0,
        absorption_rate_constant_per_hr=2.0,
        distribution_volume_l=12.0,
        clearance_l_hr=30.0,
        half_life_hours=1.0,
        protein_binding_percent=95.0,
        target_organs=["Gut"],
        therapeutic_effects={
            "Gut": {
                "gastric_acid_ph": 2.0
            }
        },
        therapeutic_index=20.0,
        toxic_concentration_mg_l=200.0
    )


# ==================== CNS DRUGS ====================

def create_sertraline(dose_mg: float = 50.0) -> DrugIntervention:
    """SSRI antidepressant."""
    return DrugIntervention(
        drug_name="Sertraline",
        drug_class="SSRI Antidepressant",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=44.0,
        absorption_rate_constant_per_hr=0.8,
        distribution_volume_l=1300.0,
        clearance_l_hr=40.0,
        half_life_hours=26.0,
        protein_binding_percent=98.0,
        target_organs=["Brain"],
        therapeutic_effects={
            "Brain": {
                "serotonin_nmol_g": 0.2
            }
        },
        therapeutic_index=15.0,
        toxic_concentration_mg_l=500.0
    )


def create_gabapentin(dose_mg: float = 300.0) -> DrugIntervention:
    """Anticonvulsant for neuropathic pain."""
    return DrugIntervention(
        drug_name="Gabapentin",
        drug_class="Anticonvulsant",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=60.0,
        absorption_rate_constant_per_hr=1.0,
        distribution_volume_l=58.0,
        clearance_l_hr=10.0,
        half_life_hours=5.0,
        protein_binding_percent=0.0,
        target_organs=["Brain"],
        therapeutic_effects={
            "Brain": {
                "gaba_umol_g": 0.5
            }
        },
        therapeutic_index=10.0,
        toxic_concentration_mg_l=100.0
    )


# ==================== PAIN MANAGEMENT ====================

def create_ibuprofen(dose_mg: float = 400.0) -> DrugIntervention:
    """NSAID for pain and inflammation."""
    return DrugIntervention(
        drug_name="Ibuprofen",
        drug_class="NSAID",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=80.0,
        absorption_rate_constant_per_hr=2.0,
        distribution_volume_l=10.0,
        clearance_l_hr=5.0,
        half_life_hours=2.0,
        protein_binding_percent=99.0,
        target_organs=["Immune System", "Kidneys"],
        therapeutic_effects={
            "Immune System": {
                "inflammatory_cytokines_pg_ml": -20.0
            }
        },
        therapeutic_index=10.0,
        toxic_concentration_mg_l=200.0
    )


def create_acetaminophen(dose_mg: float = 500.0) -> DrugIntervention:
    """Analgesic and antipyretic."""
    return DrugIntervention(
        drug_name="Acetaminophen/Paracetamol",
        drug_class="Analgesic",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=88.0,
        absorption_rate_constant_per_hr=2.5,
        distribution_volume_l=50.0,
        clearance_l_hr=20.0,
        half_life_hours=2.5,
        protein_binding_percent=20.0,
        target_organs=["Liver", "Brain"],
        therapeutic_effects={
            "Brain": {
                "pain_perception_percent": -40.0
            }
        },
        therapeutic_index=3.0,
        toxic_concentration_mg_l=150.0
    )


def create_rapamycin(dose_mg: float = 1.0) -> DrugIntervention:
    """Geroprotector (mTOR inhibitor). Aims to slow cellular senescence."""
    return DrugIntervention(
        drug_name="Rapamycin",
        drug_class="Geroprotector / mTOR Inhibitor",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=15.0,
        absorption_rate_constant_per_hr=0.5,
        distribution_volume_l=12.0,
        clearance_l_hr=0.5,
        half_life_hours=60.0,
        protein_binding_percent=92.0,
        target_organs=["Immune System", "Muscles", "Brain"],
        therapeutic_effects={
            "Immune System": {
                "systemic_inflammation": -0.2, # Reduces Inflammaging
                "macrophage_activity_percent": 10.0 # Autophagy stimulation
            },
            "Muscles": {
                "protein_synthesis_percent": 5.0 # Quality control
            },
            "Brain": {
                "neuroinflammation": -0.1
            }
        },
        therapeutic_index=8.0,
        toxic_concentration_mg_l=0.03
    )


# ==================== GERIATRIC/PSYCHIATRIC DRUGS ====================

def create_lorazepam(dose_mg: float = 1.0) -> DrugIntervention:
    """Benzodiazepine for anxiety/sedation. Potentially inappropriate for elderly."""
    return DrugIntervention(
        drug_name="Lorazepam",
        drug_class="Benzodiazepine",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=90.0,
        absorption_rate_constant_per_hr=1.5,
        distribution_volume_l=91.0,
        clearance_l_hr=4.0,
        half_life_hours=12.0,
        protein_binding_percent=85.0,
        target_organs=["Brain"],
        therapeutic_effects={
            "Brain": {
                "gaba_umol_g": 0.3,
                "processing_speed_ms": 50.0 # Slowing
            }
        },
        therapeutic_index=10.0,
        toxic_concentration_mg_l=0.5
    )


def create_zolpidem(dose_mg: float = 5.0) -> DrugIntervention:
    """Sedative-hypnotic for insomnia. High fall risk in elderly."""
    return DrugIntervention(
        drug_name="Zolpidem",
        drug_class="Z-drug Sedative",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=70.0,
        absorption_rate_constant_per_hr=2.0,
        distribution_volume_l=35.0,
        clearance_l_hr=15.0,
        half_life_hours=2.5,
        protein_binding_percent=92.0,
        target_organs=["Brain"],
        therapeutic_effects={
            "Brain": {
                "gaba_umol_g": 0.4
            }
        },
        therapeutic_index=15.0,
        toxic_concentration_mg_l=1.0
    )


def create_amitriptyline(dose_mg: float = 25.0) -> DrugIntervention:
    """TCA antidepressant. Highly anticholinergic."""
    return DrugIntervention(
        drug_name="Amitriptyline",
        drug_class="Tricyclic Antidepressant",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=45.0,
        absorption_rate_constant_per_hr=1.0,
        distribution_volume_l=1000.0,
        clearance_l_hr=40.0,
        half_life_hours=20.0,
        protein_binding_percent=95.0,
        target_organs=["Brain", "Heart"],
        therapeutic_effects={
            "Brain": {
                "serotonin_nmol_g": 0.1,
                "acetylcholine_nmol_g": -0.1 # Anticholinergic
            }
        },
        therapeutic_index=5.0,
        toxic_concentration_mg_l=1.0
    )


def create_omeprazole(dose_mg: float = 20.0) -> DrugIntervention:
    """Proton Pump Inhibitor for reflux. Long-term use in elderly can cause fractures/infections."""
    return DrugIntervention(
        drug_name="Omeprazole",
        drug_class="PPI",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=40.0,
        absorption_rate_constant_per_hr=1.2,
        distribution_volume_l=25.0,
        clearance_l_hr=30.0,
        half_life_hours=1.0,
        protein_binding_percent=95.0,
        target_organs=["Gut"],
        therapeutic_effects={
            "Gut": {
                "gastric_acid_ph": 4.5,
                "nutrient_absorption_percent": -5.0 # Side effect: Malabsorption
            },
            "Bones": {
                "bone_mineral_density_percent": -0.5 # Long term risk
            }
        },
        therapeutic_index=20.0,
        toxic_concentration_mg_l=2.0
    )


def create_furosemide(dose_mg: float = 40.0) -> DrugIntervention:
    """Loop Diuretic. Used for edema, often secondary to Amlodipine."""
    return DrugIntervention(
        drug_name="Furosemide",
        drug_class="Loop Diuretic",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=60.0,
        absorption_rate_constant_per_hr=1.5,
        distribution_volume_l=8.0,
        clearance_l_hr=15.0,
        half_life_hours=1.5,
        protein_binding_percent=98.0,
        target_organs=["Kidneys", "Heart"],
        therapeutic_effects={
            "Kidneys": {
                "fluid_retention_ml": -500.0,
                "gfr_ml_min": -5.0 # Acute stress
            },
            "Heart": {
                "preload_mmhg": -5.0
            }
        },
        therapeutic_index=15.0,
        toxic_concentration_mg_l=3.0
    )


def create_alprazolam(dose_mg: float = 0.5) -> DrugIntervention:
    """Benzodiazepine (Xanax). High fall/delirium risk in elderly."""
    return DrugIntervention(
        drug_name="Alprazolam",
        drug_class="Benzodiazepine",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=85.0,
        absorption_rate_constant_per_hr=2.0,
        distribution_volume_l=70.0,
        clearance_l_hr=6.0,
        half_life_hours=11.0,
        protein_binding_percent=80.0,
        target_organs=["Brain"],
        therapeutic_effects={
            "Brain": {
                "gaba_umol_g": 0.5,
                "processing_speed_ms": 300.0 # Extreme slowing/delirium risk
            }
        },
        therapeutic_index=15.0,
        toxic_concentration_mg_l=0.1
    )


def create_amlodipine(dose_mg: float = 5.0) -> DrugIntervention:
    """Calcium Channel Blocker for hypertension. Side effect: Peripheral edema."""
    return DrugIntervention(
        drug_name="Amlodipine",
        drug_class="CCB",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=64.0,
        absorption_rate_constant_per_hr=0.5,
        distribution_volume_l=1500.0,
        clearance_l_hr=25.0,
        half_life_hours=40.0,
        protein_binding_percent=97.0,
        target_organs=["Heart", "Skin"],
        therapeutic_effects={
            "Heart": {
                "afterload_mmhg": -15.0
            },
            "Skin": {
                "peripheral_edema_index": 2.5 # Significant edema
            }
        },
        therapeutic_index=12.0,
        toxic_concentration_mg_l=0.2
    )


# ==================== DRUG LIBRARY REGISTRY ====================

DRUG_LIBRARY = {
    "lisinopril": "create_ace_inhibitor",
    "amlodipine": create_amlodipine,
    "metoprolol": "create_beta_blocker",
    "atorvastatin": create_atorvastatin,
    "furosemide": create_furosemide,
    "warfarin": create_warfarin,
    "digoxin": create_digoxin,
    "metformin": "create_metformin",
    "insulin": create_insulin_rapid,
    "levothyroxine": create_levothyroxine,
    "omeprazole": create_omeprazole,
    "sertraline": create_sertraline,
    "gabapentin": create_gabapentin,
    "ibuprofen": create_ibuprofen,
    "acetaminophen": create_acetaminophen,
    "lorazepam": create_lorazepam,
    "zolpidem": create_zolpidem,
    "amitriptyline": create_amitriptyline,
    "rapamycin": create_rapamycin,
    "omeprazole": create_omeprazole,
    "furosemide": create_furosemide,
    "alprazolam": create_alprazolam,
    "amlodipine": create_amlodipine,
    "coq10": "create_coq10_supplement"
}


def get_drug(drug_name: str, dose: float = None) -> DrugIntervention:
    """Get drug from library by name."""
    drug_name_lower = drug_name.lower()
    
    if drug_name_lower not in DRUG_LIBRARY:
        raise ValueError(f"Drug '{drug_name}' not in library.")
    
    creator = DRUG_LIBRARY[drug_name_lower]
    
    if isinstance(creator, str):
        from . import drug_intervention
        creator_func = getattr(drug_intervention, creator)
        return creator_func(dose) if dose is not None else creator_func()
    else:
        return creator(dose) if dose is not None else creator()


def list_available_drugs() -> List[str]:
    """List all available drugs in library."""
    return sorted(DRUG_LIBRARY.keys())
