"""
Drug Intervention System.

Models pharmacokinetics and pharmacodynamics for drug interventions.
Integrates with Medical NanoCERN for safety validation.
"""

from dataclasses import dataclass, field
from typing import Dict, List, Any, Optional
from enum import Enum


class RouteOfAdministration(Enum):
    """Routes of drug administration."""
    ORAL = "oral"
    IV = "intravenous"
    IM = "intramuscular"
    SC = "subcutaneous"
    TRANSDERMAL = "transdermal"
    INHALATION = "inhalation"


@dataclass
class DrugIntervention:
    """
    Complete drug intervention model with PK/PD.
    
    Pharmacokinetics (ADME):
    - Absorption
    - Distribution
    - Metabolism
    - Excretion
    
    Pharmacodynamics:
    - Receptor binding
    - Organ effects
    - Therapeutic effects
    - Adverse effects
    """
    
    # Drug identification
    drug_name: str
    drug_class: str
    dose_mg: float
    route: RouteOfAdministration
    
    # Pharmacokinetics
    bioavailability_percent: float = 100.0  # F
    absorption_rate_constant_per_hr: float = 1.0  # ka
    distribution_volume_l: float = 50.0  # Vd
    clearance_l_hr: float = 5.0  # CL
    half_life_hours: float = 4.0  # t1/2
    protein_binding_percent: float = 90.0
    
    # Current state
    plasma_concentration_mg_l: float = 0.0
    time_since_administration_hr: float = 0.0
    
    # Pharmacodynamics
    target_organs: List[str] = field(default_factory=list)
    therapeutic_effects: Dict[str, float] = field(default_factory=dict)
    adverse_effects: Dict[str, float] = field(default_factory=dict)
    
    # Safety
    therapeutic_index: float = 10.0  # LD50 / ED50
    toxic_concentration_mg_l: float = 100.0
    
    def calculate_plasma_concentration(self, time_hr: float) -> float:
        """
        Calculate plasma concentration at given time.
        
        Uses one-compartment model with first-order kinetics.
        
        Args:
            time_hr: Time since administration (hours)
            
        Returns:
            Plasma concentration (mg/L)
        """
        if self.route == RouteOfAdministration.IV:
            # IV: C(t) = (Dose/Vd) * e^(-k*t)
            k = 0.693 / self.half_life_hours
            C0 = self.dose_mg / self.distribution_volume_l
            return C0 * (2.718 ** (-k * time_hr))
        else:
            # Oral: C(t) = (F*Dose*ka)/(Vd*(ka-k)) * (e^(-k*t) - e^(-ka*t))
            k = 0.693 / self.half_life_hours
            ka = self.absorption_rate_constant_per_hr
            F = self.bioavailability_percent / 100.0
            
            if abs(ka - k) < 0.001:  # Avoid division by zero
                return 0.0
            
            C = (F * self.dose_mg * ka) / (self.distribution_volume_l * (ka - k))
            C *= ((2.718 ** (-k * time_hr)) - (2.718 ** (-ka * time_hr)))
            return max(0, C)
    
    def update_state(self, time_step_hr: float):
        """
        Update drug state for time step.
        
        Args:
            time_step_hr: Time step (hours)
        """
        self.time_since_administration_hr += time_step_hr
        self.plasma_concentration_mg_l = self.calculate_plasma_concentration(
            self.time_since_administration_hr
        )
    
    def is_therapeutic(self) -> bool:
        """Check if concentration is in therapeutic range."""
        # Simplified: therapeutic range is 10-90% of toxic concentration
        therapeutic_min = self.toxic_concentration_mg_l * 0.1
        therapeutic_max = self.toxic_concentration_mg_l * 0.9
        return therapeutic_min <= self.plasma_concentration_mg_l <= therapeutic_max
    
    def is_toxic(self) -> bool:
        """Check if concentration is toxic."""
        return self.plasma_concentration_mg_l > self.toxic_concentration_mg_l
    
    def apply_to_body(self, body_state) -> Dict[str, Any]:
        """
        Apply drug effects to body state.
        
        Args:
            body_state: HumanBodyState instance
            
        Returns:
            Dictionary of effects applied
        """
        effects_applied = {}
        
        # Only apply effects if in therapeutic range
        if not self.is_therapeutic():
            return effects_applied
        
        # Apply therapeutic effects to target organs
        for organ_name, effect_dict in self.therapeutic_effects.items():
            if organ_name in body_state.get_all_organs():
                organ = body_state.get_all_organs()[organ_name]
                organ.apply_effects(effect_dict)
                effects_applied[organ_name] = effect_dict
        
        return effects_applied


# ==================== DRUG LIBRARY ====================

def create_ace_inhibitor(dose_mg: float = 10.0) -> DrugIntervention:
    """Create ACE inhibitor (e.g., lisinopril) intervention."""
    return DrugIntervention(
        drug_name="ACE Inhibitor (Lisinopril)",
        drug_class="Antihypertensive",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=25.0,
        absorption_rate_constant_per_hr=0.8,
        distribution_volume_l=70.0,
        clearance_l_hr=5.0,
        half_life_hours=12.0,
        protein_binding_percent=10.0,
        target_organs=["Heart", "Kidneys"],
        therapeutic_effects={
            "Heart": {
                "afterload_mmhg": -20.0,  # Reduce afterload
                "heart_rate_bpm": -5.0
            },
            "Kidneys": {
                "renin_secretion_ng_ml_hr": -0.5
            }
        },
        therapeutic_index=10.0,
        toxic_concentration_mg_l=50.0
    )


def create_beta_blocker(dose_mg: float = 50.0) -> DrugIntervention:
    """Create beta blocker (e.g., metoprolol) intervention."""
    return DrugIntervention(
        drug_name="Beta Blocker (Metoprolol)",
        drug_class="Antihypertensive",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=50.0,
        absorption_rate_constant_per_hr=1.5,
        distribution_volume_l=250.0,
        clearance_l_hr=50.0,
        half_life_hours=3.5,
        protein_binding_percent=12.0,
        target_organs=["Heart"],
        therapeutic_effects={
            "Heart": {
                "heart_rate_bpm": -20.0,  # Reduce HR
                "contractility_percent": -15.0  # Reduce contractility
            }
        },
        therapeutic_index=8.0,
        toxic_concentration_mg_l=40.0
    )


def create_metformin(dose_mg: float = 1000.0) -> DrugIntervention:
    """Create metformin intervention for diabetes."""
    return DrugIntervention(
        drug_name="Metformin",
        drug_class="Antidiabetic",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=50.0,
        absorption_rate_constant_per_hr=2.0,
        distribution_volume_l=300.0,
        clearance_l_hr=30.0,
        half_life_hours=6.0,
        protein_binding_percent=0.0,
        target_organs=["Liver", "Muscles"],
        therapeutic_effects={
            "Liver": {
                "glucose_production_mg_dl_hr": -50.0  # Reduce hepatic glucose output
            },
            "Muscles": {
                "glucose_uptake_percent": 20.0  # Increase muscle glucose uptake
            }
        },
        therapeutic_index=15.0,
        toxic_concentration_mg_l=100.0
    )


def create_coq10_supplement(dose_mg: float = 200.0) -> DrugIntervention:
    """Create CoQ10 supplement intervention."""
    return DrugIntervention(
        drug_name="CoQ10",
        drug_class="Supplement",
        dose_mg=dose_mg,
        route=RouteOfAdministration.ORAL,
        bioavailability_percent=10.0,
        absorption_rate_constant_per_hr=0.5,
        distribution_volume_l=100.0,
        clearance_l_hr=2.0,
        half_life_hours=33.0,
        protein_binding_percent=95.0,
        target_organs=["Heart", "Brain", "Muscles"],
        therapeutic_effects={
            "Heart": {
                "atp_production_percent": 20.0,
                "mitochondrial_density_percent": 10.0
            },
            "Brain": {
                "mitochondrial_function_percent": 15.0,
                "atp_production_percent": 15.0
            },
            "Muscles": {
                "atp_stores_percent": 15.0
            }
        },
        therapeutic_index=50.0,  # Very safe
        toxic_concentration_mg_l=1000.0
    )


# ==================== MEDICAL NANOCERN INTEGRATION ====================

def validate_drug_safety(drug: DrugIntervention, body_state) -> Dict[str, Any]:
    """
    Validate drug safety using Medical NanoCERN principles.
    
    Args:
        drug: DrugIntervention instance
        body_state: HumanBodyState instance
        
    Returns:
        Safety validation result
    """
    safety_result = {
        "is_safe": True,
        "nkus_generated": [],
        "warnings": [],
        "contraindications": []
    }
    
    # Check organ function envelopes
    organs = body_state.get_all_organs()
    
    # Example: ACE inhibitors contraindicated in severe renal impairment
    if "ACE Inhibitor" in drug.drug_name:
        if organs["Kidneys"].gfr_ml_min < 30:
            safety_result["is_safe"] = False
            safety_result["contraindications"].append({
                "type": "organ_function_violation",
                "organ": "Kidneys",
                "parameter": "GFR",
                "value": organs["Kidneys"].gfr_ml_min,
                "threshold": 30.0,
                "severity": "major",
                "reason": "ACE inhibitors contraindicated in severe renal impairment (GFR <30)"
            })
    
    # Example: Beta blockers contraindicated in severe bradycardia
    if "Beta Blocker" in drug.drug_name:
        if organs["Heart"].heart_rate_bpm < 50:
            safety_result["is_safe"] = False
            safety_result["contraindications"].append({
                "type": "hemodynamic_violation",
                "organ": "Heart",
                "parameter": "heart_rate_bpm",
                "value": organs["Heart"].heart_rate_bpm,
                "threshold": 50.0,
                "severity": "critical",
                "reason": "Beta blockers contraindicated in bradycardia (HR <50)"
            })
    
    # Example: Metformin contraindicated in renal impairment
    if "Metformin" in drug.drug_name:
        if organs["Kidneys"].gfr_ml_min < 45:
            safety_result["is_safe"] = False
            safety_result["contraindications"].append({
                "type": "organ_function_violation",
                "organ": "Kidneys",
                "parameter": "GFR",
                "value": organs["Kidneys"].gfr_ml_min,
                "threshold": 45.0,
                "severity": "major",
                "reason": "Metformin contraindicated in moderate-severe renal impairment (risk of lactic acidosis)"
            })
    
    return safety_result
