"""
Drug Discovery Engine: The "Thinking Machine" of Physiology.

Automates the search for drug candidates and dosage optimization 
to restore homeostatic balance in complex disease states.
"""

import copy
from typing import Dict, List, Any, Tuple
from body_state_model import HumanBodyState, BodySystemOrchestrator
from drugs.drug_library import get_drug, list_available_drugs
from safety_reactor import SafetyReactor

class DrugDiscoveryEngine:
    """Autonomous engine for rational drug discovery and optimization."""
    
    def __init__(self):
        self.reactor = SafetyReactor()
        
    def find_best_intervention(self, patient_state: HumanBodyState) -> Dict[str, Any]:
        """
        Screen all available drugs with high-precision rational reasoning.
        """
        print(f"\n[THINK] BEGINNING PATHOPHYSIOLOGICAL ANALYSIS...")
        initial_h_data = patient_state.get_overall_homeostasis()
        baseline_score = initial_h_data["overall_homeostasis_score"]
        imbalanced_organs = initial_h_data["imbalanced_organs"]
        
        # Identify specific parameter deficits
        print(f"[THINK] Vital Sign Analysis: {len(imbalanced_organs)} systems compromised.")
        for organ_name in imbalanced_organs:
            organ = patient_state.get_all_organs()[organ_name]
            score = organ.calculate_homeostasis_score()
            print(f"  • {organ_name:15} | Score: {score:5.1f}% | State: {organ.state_classification}")
        
        print(f"[THINK] Scanning library for homeostatic reset candidates...")
        
        candidates = list_available_drugs()
        results = []
        
        for drug_name in candidates:
            try:
                drug = get_drug(drug_name)
            except:
                continue
                
            # 1. Safety Filter (Medical NanoCERN)
            safety_check = self.reactor.validate_drug(drug, patient_state)
            if not safety_check["is_safe"]:
                print(f"  [X] {drug_name:15} | REJECTED: Safety Envelope Violation")
                continue
                
            # 2. Virtual Simulation (Short window to isolate drug effect)
            v_patient = copy.deepcopy(patient_state)
            
            # Apply therapeutic effects
            applicable = False
            for organ_name, effects in drug.therapeutic_effects.items():
                target_key = organ_name.lower().replace(" ", "_").replace("systemic_", "").replace("_system", "")
                if hasattr(v_patient, target_key):
                    target_organ = getattr(v_patient, target_key)
                    target_organ.apply_effects(effects)
                    applicable = True
            
            if not applicable:
                continue

            # Measure immediate homeostatic delta (without simulation drift)
            new_h_data = v_patient.get_overall_homeostasis()
            new_score = new_h_data["overall_homeostasis_score"]
            delta = new_score - baseline_score
            
            results.append({
                "drug_name": drug_name,
                "homeostasis_gain": delta,
                "final_score": new_score,
                "targets": list(drug.therapeutic_effects.keys())
            })
            
            if delta > 0.05:
                print(f"  [+] {drug_name:15} | BENEFICIAL: +{delta:5.2f}% stabilization predicted.")
            else:
                print(f"  [-] {drug_name:15} | LOW IMPACT: {delta:5.2f}% gain.")
            
        # Sort by gain
        results.sort(key=lambda x: x["homeostasis_gain"], reverse=True)
        
        if results and results[0]["homeostasis_gain"] > 0:
            winner = results[0]
            print(f"\n[VERDICT] Optimal Therapeutic Intervention: {winner['drug_name'].upper()}")
            print(f"[VERDICT] Mechanism: Targeted homeostatic reset of {', '.join(winner['targets'])}.")
            return winner
        return None

    def optimize_dosage(self, drug_name: str, patient_state: HumanBodyState) -> float:
        """
        Gradient-based (simulated) dosage optimization.
        Finds the maximum dose before toxicity or side-effects outweigh benefit.
        """
        print(f"\n[THINKING] Optimizing dosage for {drug_name}...")
        best_dose = 0.0
        best_gain = -999.0
        
        # Test range of doses
        for dose in [5.0, 10.0, 20.0, 40.0, 80.0, 160.0]:
            try:
                drug = get_drug(drug_name, dose)
            except:
                continue
                
            safety_check = self.reactor.validate_drug(drug, patient_state)
            if not safety_check["is_safe"]:
                break # Reached safety limit
                
            v_patient = copy.deepcopy(patient_state)
            v_orchestrator = BodySystemOrchestrator(v_patient)
            
            # Simplified PK/PD apply
            # (In production, this is a nested simulation loop)
            for organ_name, effects in drug.therapeutic_effects.items():
                if hasattr(v_patient, organ_name.lower().replace(" ", "_")):
                    organ = getattr(v_patient, organ_name.lower().replace(" ", "_"))
                    # Scale effects by dose relative to standard
                    scaled_effects = {k: v * (dose/20.0) for k, v in effects.items()} 
                    organ.apply_effects(scaled_effects)
            
            v_orchestrator.step()
            homeostasis = v_patient.get_overall_homeostasis()["overall_homeostasis_score"]
            
            if homeostasis > best_gain:
                best_gain = homeostasis
                best_dose = dose
                
        print(f"  Optimal Dose Found: {best_dose}mg (Max Homeostasis: {best_gain:.1f}%)")
        return best_dose
