"""
Deprescribing Engine: The Inverse Discovery Machine.

Identifies and recommends removal of medications that have negative 
homeostatic ROI in complex geriatric patients.
"""

import copy
from typing import Dict, List, Any
from body_state_model import HumanBodyState, BodySystemOrchestrator
from drugs.drug_library import get_drug

class DeprescribingEngine:
    """Autonomous engine for simplifying polypharmacy."""
    
    def __init__(self):
        pass
        
    def analyze_polypharmacy(self, patient_state: HumanBodyState, active_meds: List[Dict[str, float]]) -> Dict[str, Any]:
        """
        Subtractive screening to identify homeostatic gains from medication removal.
        """
        print(f"\n[DEPRESCRIBE] ANALYZING POLYPHARMACY REGIMEN ({len(active_meds)} drugs)...")
        
        # 1. Establish Baseline (Patient with ALL meds)
        baseline_patient = copy.deepcopy(patient_state)
        self._apply_full_regimen(baseline_patient, active_meds)
        
        baseline_h = baseline_patient.get_overall_homeostasis()
        baseline_score = baseline_h["overall_homeostasis_score"]
        
        print(f"[DEPRESCRIBE] Baseline Homeostasis (on all meds): {baseline_score:.1f}%")
        print(f"[DEPRESCRIBE] Searching for homeostatic rebound candidates...")
        
        recommendations = []
        
        # 2. Iterative Subtraction
        for i, med_to_remove in enumerate(active_meds):
            drug_name = med_to_remove["name"]
            
            # Create a virtual world where this specific drug is REMOVED
            v_patient = copy.deepcopy(patient_state)
            reduced_regimen = [m for j, m in enumerate(active_meds) if i != j]
            
            self._apply_full_regimen(v_patient, reduced_regimen)
            
            # Step simulation to stabilize
            v_orchestrator = BodySystemOrchestrator(v_patient)
            v_orchestrator.run_simulation(duration_seconds=10.0)
            
            new_h = v_patient.get_overall_homeostasis()
            new_score = new_h["overall_homeostasis_score"]
            delta = new_score - baseline_score
            
            # Analyze impact
            if delta > 0.01:
                status = "RECOMB RECOMMENDATION"
                print(f"  [!] {drug_name:15} | REBOUND: +{delta:5.2f}% | (Before: {baseline_score:.1f}, After: {new_score:.1f})")
            elif delta > -0.01:
                status = "CONSIDER REMOVAL"
                print(f"  [-] {drug_name:15} | NEUTRAL:  {delta:5.2f}% | (Before: {baseline_score:.1f}, After: {new_score:.1f})")
            else:
                status = "KEEP"
                print(f"  [X] {drug_name:15} | LOSS:     {delta:5.2f}% | (Before: {baseline_score:.1f}, After: {new_score:.1f})")
                
            recommendations.append({
                "drug_name": drug_name,
                "homeostasis_rebound": delta,
                "status": status
            })
            
        # Sort recommendations by rebound gain
        recommendations.sort(key=lambda x: x["homeostasis_rebound"], reverse=True)
        return {
            "baseline_score": baseline_score,
            "recommendations": recommendations,
            "potential_clean_score": baseline_score + sum(r["homeostasis_rebound"] for r in recommendations if r["homeostasis_rebound"] > 0)
        }

    def global_cleanup(self, patient_state: HumanBodyState, active_meds: List[Dict[str, float]]) -> Dict[str, Any]:
        """Simulate removal of all potentially inappropriate medications."""
        print(f"\n[THINK] SIMULATING GLOBAL REGIMEN CLEAN-UP...")
        
        # 1. Baseline
        baseline_p = copy.deepcopy(patient_state)
        self._apply_full_regimen(baseline_p, active_meds)
        baseline_score = baseline_p.get_overall_homeostasis()["overall_homeostasis_score"]
        
        # 2. Identify PIMs via Beers Criteria logic + Homeostatic noise
        # (For this demo, we assume PIMs are Omeprazole, Alprazolam, Furosemide, Amlodipine)
        pims = ["Omeprazole", "Alprazolam", "Furosemide", "Amlodipine"]
        clean_regimen = [m for m in active_meds if m["name"] not in pims]
        
        clean_p = copy.deepcopy(patient_state)
        self._apply_full_regimen(clean_p, clean_regimen)
        
        # Stabilization simulation
        v_orchestrator = BodySystemOrchestrator(clean_p)
        v_orchestrator.run_simulation(duration_seconds=20.0)
        
        clean_score = clean_p.get_overall_homeostasis()["overall_homeostasis_score"]
        
        return {
            "initial": baseline_score,
            "final": clean_score,
            "removed": pims,
            "rebound": clean_score - baseline_score
        }

    def _apply_full_regimen(self, patient: HumanBodyState, meds: List[Dict[str, float]]):
        """Helper to apply a list of meds to a patient state."""
        for med in meds:
            try:
                drug = get_drug(med["name"], med["dose"])
                # Apply therapeutic effects directly for simulation
                for organ_name, effects in drug.therapeutic_effects.items():
                    target_key = organ_name.lower().replace(" ", "_").replace("systemic_", "").replace("_system", "")
                    if hasattr(patient, target_key):
                        target_organ = getattr(patient, target_key)
                        target_organ.apply_effects(effects)
            except:
                continue
