# Complete State-Based Physiological Model

A comprehensive digital twin of human physiology for mechanistic drug discovery and disease modeling.

## Overview

This system models the human body as a network of interconnected state machines, where each organ has:
- **Basal state** (normal physiology)
- **State variables** (numerical parameters)
- **State transitions** (disease states)
- **Inter-organ connections** (cascading effects)

## Current Implementation

### Implemented Organs (6/12)

1. **Heart** (`organs/heart.py`)
   - Hemodynamics (HR, SV, CO, BP)
   - Electrical conduction
   - Metabolic function
   - Disease states: Heart failure, hypertrophy, ischemia, arrhythmia

2. **Lungs** (`organs/lungs.py`)
   - Ventilation mechanics
   - Gas exchange
   - Perfusion
   - Disease states: COPD, asthma, pneumonia

3. **Liver** (`organs/liver.py`)
   - Synthetic function (albumin, clotting factors)
   - Metabolic function (glucose, lipids)
   - Detoxification (P450 enzymes)
   - Disease states: Cirrhosis, fatty liver, hepatitis

4. **Kidneys** (`organs/kidneys.py`)
   - Filtration (GFR)
   - Tubular function
   - Endocrine (renin, EPO, vitamin D)
   - Disease states: CKD stages 1-5, AKI

5. **Brain** (`organs/brain.py`)
   - Perfusion and metabolism
   - Neurotransmitter systems
   - Cognitive function
   - Disease states: Alzheimer's, Parkinson's, stroke

6. **Pancreas** (`organs/pancreas.py`)
   - Endocrine (insulin, glucagon)
   - Exocrine (digestive enzymes)
   - Disease states: Type 1/2 diabetes, pancreatitis

### Core Architecture

- **`organs/base_organ.py`**: Abstract base class for all organs
- **`body_state_model.py`**: Main orchestrator with `HumanBodyState` and `BodySystemOrchestrator`
- **`demo_scenarios.py`**: 8 demonstration scenarios

## Quick Start

### 1. Basic Usage

```python
from body_state_model import HumanBodyState, BodySystemOrchestrator

# Create healthy body
body = HumanBodyState()
orchestrator = BodySystemOrchestrator(body)

# Check initial state
print(body.get_state_summary())

# Run simulation
orchestrator.run_simulation(duration_seconds=60.0, verbose=True)
```

### 2. Disease Modeling

```python
# Induce heart failure
body.heart.transition_to_heart_failure("ischemic")

# Observe cascade effects
orchestrator.run_simulation(60.0)

print(f"Cardiac output: {body.heart.cardiac_output_l_min:.2f} L/min")
print(f"Kidney GFR: {body.kidneys.gfr_ml_min:.1f} mL/min")
print(f"Brain perfusion: {body.brain.cerebral_blood_flow_ml_min:.0f} mL/min")
```

### 3. Run Demonstrations

```bash
# Run all 8 demonstration scenarios
python3 demo_scenarios.py

# Or run basic examples
python3 body_state_model.py
```

## Demonstration Scenarios

1. **Healthy Aging** - Baseline physiological state
2. **Heart Failure Cascade** - Cardiac dysfunction → kidney/brain effects
3. **COPD Hypoxia** - Respiratory failure → cardiac/neurological compensation
4. **Liver Cirrhosis** - Hepatic failure → encephalopathy, coagulopathy
5. **CKD** - Renal failure → anemia, bone disease, volume overload
6. **Type 1 Diabetes** - Insulin deficiency → hyperglycemia
7. **Acute Stroke** - Brain injury → consciousness impairment
8. **Alzheimer's Disease** - Neurodegeneration → cognitive decline

## Key Features

### ✅ Implemented

- 6 complete organ state machines
- 20+ disease transitions
- Inter-organ communication network
- Homeostasis scoring system
- Time-step simulation engine
- Disease cascade modeling

### 🚧 In Progress

- Remaining 6 organs (gut, immune, endocrine, muscles, bones, skin)
- Drug intervention modeling
- Integration with Medical NanoCERN safety system
- Enhanced visualization (React dashboard)

## Architecture

```
body_state_model.py
├── HumanBodyState
│   ├── heart: HeartState
│   ├── lungs: LungState
│   ├── liver: LiverState
│   ├── kidneys: KidneyState
│   ├── brain: BrainState
│   └── pancreas: PancreasState
│
└── BodySystemOrchestrator
    ├── propagate_state_changes()
    ├── step()
    └── run_simulation()
```

## Inter-Organ Connections

| From | To | Effect |
|------|-----|--------|
| Heart | Kidneys | Cardiac output → Renal perfusion |
| Heart | Brain | Cardiac output → Cerebral perfusion |
| Heart | Liver | Cardiac output → Hepatic perfusion |
| Lungs | Heart | O2 saturation → Myocardial oxygen delivery |
| Kidneys | Heart | Fluid retention → Preload |
| Liver | Brain | Ammonia clearance → Encephalopathy |
| Pancreas | Liver | Insulin/glucagon → Glucose production |

## Disease Transitions

Each organ implements multiple disease states:

```python
# Example: Heart failure
body.heart.transition_to_heart_failure("ischemic")
# → Ejection fraction ↓42%
# → Cardiac output ↓30%
# → BNP ↑900%

# Example: CKD Stage 4
body.kidneys.transition_to_ckd(stage=4)
# → GFR ↓78%
# → Creatinine ↑400%
# → EPO ↓78%
```

## Homeostasis Scoring

Each organ calculates deviation from basal state:

```python
homeostasis = body.get_overall_homeostasis()
# {
#   "overall_homeostasis_score": 85.7,
#   "organ_scores": {
#     "Heart": 31.6,
#     "Lungs": 92.4,
#     ...
#   },
#   "imbalanced_organs": ["Heart"]
# }
```

## Next Steps

### Phase 3: Remaining Organs
- Implement gut, immune, endocrine, muscles, bones, skin

### Phase 4: Drug Intervention
- Create `DrugIntervention` class
- Model pharmacokinetics (ADME)
- Model pharmacodynamics (receptor binding)
- Integrate with Medical NanoCERN

### Phase 5: Visualization
- Enhance React/TSX dashboard
- Add all 12 organ systems
- Real-time monitoring
- Disease scenario selector

## Integration with Medical NanoCERN

This physiological model will integrate with the Medical NanoCERN safety system:

1. **Body state → Medical NanoCERN**: Use organ states as input to safety validation
2. **Drug intervention → Safety check**: Validate drug effects against safety envelopes
3. **NKU generation**: Generate medical NKUs when organ states violate safety boundaries

## File Structure

```
stem/
├── body_state_model.py          # Main orchestrator
├── demo_scenarios.py            # 8 demonstration scenarios
├── organs/
│   ├── __init__.py
│   ├── base_organ.py           # Abstract base class
│   ├── heart.py                # Cardiac state machine
│   ├── lungs.py                # Pulmonary state machine
│   ├── liver.py                # Hepatic state machine
│   ├── kidneys.py              # Renal state machine
│   ├── brain.py                # Neurological state machine
│   └── pancreas.py             # Pancreatic state machine
├── diseases/                    # (To be implemented)
├── drugs/                       # (To be implemented)
├── simulation/                  # (To be implemented)
└── tests/                       # (To be implemented)
```

## Example Output

```
=== Human Body State (t=60.0s) ===
Overall Homeostasis: 85.7%
Health Status: HEALTHY

Organ Status:
  Heart: FAILING (31.6%)
  Lungs: NORMAL (92.4%)
  Liver: NORMAL (96.7%)
  Kidneys: NORMAL (94.8%)
  Brain: NORMAL (98.7%)
  Pancreas: NORMAL (100.0%)

⚠️  Imbalanced Organs: Heart

Cascade effects observed:
  - Cardiac output: 3.50 L/min (↓30%)
  - Kidney GFR: 58.33 mL/min
  - Brain perfusion: 525 ml/min
```

## Contributing

This is a research project for mechanistic drug discovery. Future extensions:
- Additional organ systems
- More disease states
- Drug-drug interactions
- Temporal disease progression
- Probabilistic risk modeling

## License

Research use only. Not approved for clinical use.

## References

- Organ parameter values validated against medical literature
- Disease transitions based on clinical guidelines
- Inter-organ effects from physiology textbooks
