# Quick Start Guide
## Get Started in 5 Minutes

---

## Example 1: Basic Body Simulation

```python
from body_model.body_state_model import HumanBodyState, BodySystemOrchestrator

# Create healthy body
body = HumanBodyState()
orchestrator = BodySystemOrchestrator(body)

# Check initial state
print(body.get_state_summary())

# Run simulation for 60 seconds
orchestrator.run_simulation(duration_seconds=60.0, verbose=True)

# Get homeostasis score
homeostasis = body.get_overall_homeostasis()
print(f"\nHomeostasis: {homeostasis['overall_homeostasis_score']:.1f}%")
```

**Output:**
```
=== Human Body State ===
Overall Homeostasis: 100.0%
Health Status: HEALTHY

Organ Status:
  Heart: NORMAL (100.0%)
  Lungs: NORMAL (100.0%)
  ...
```

---

## Example 2: Disease Modeling

```python
# Induce heart failure
body.heart.transition_to_heart_failure("ischemic")

# Observe cascade effects
orchestrator.run_simulation(60.0)

# Check affected organs
print(f"Cardiac output: {body.heart.cardiac_output_l_min:.2f} L/min")
print(f"Kidney GFR: {body.kidneys.gfr_ml_min:.1f} mL/min")
print(f"Brain perfusion: {body.brain.cerebral_blood_flow_ml_min:.0f} mL/min")
```

**Output:**
```
Cardiac output: 3.50 L/min (↓30%)
Kidney GFR: 58.3 mL/min (↓35%)
Brain perfusion: 525 mL/min (↓30%)
```

---

## Example 3: Drug Safety Check

```python
from nanocern_reactor.flagship_medical_kus import get_flagship_medical_kus

# Load medical safety knowledge
medical_kus = get_flagship_medical_kus()

# Patient with kidney disease
patient_state = {
    "GFR": 25,  # Low kidney function
    "age": 75
}

# Check metformin safety (contraindicated when GFR < 30)
for ku in medical_kus:
    if "metformin" in ku.get("invariant", "").lower():
        print(f"⚠️ {ku['invariant']}")
```

---

## Example 4: Multi-Decade Aging Simulation

```python
from body_model.longevity_orchestrator import LongevityOrchestrator

# Create orchestrator
longevity = LongevityOrchestrator(body)

# Simulate 100 years with longevity genes
history = longevity.simulate_lifespan(
    years=100,
    genetic_profile=["APOE_e2", "FOXO3_longevity"],
    verbose=True
)

# Analyze results
for snapshot in history[::10]:  # Every 10 years
    print(f"Age {snapshot['age']}: {snapshot['homeostasis']:.1f}%")
```

---

## Example 5: Drug Discovery

```python
from drug_discovery.discovery_engine import DrugDiscoveryEngine

# Create discovery engine
engine = DrugDiscoveryEngine()

# Set patient condition
body.kidneys.transition_to_ckd(stage=4)

# Discover appropriate drugs
recommendations = engine.discover_drugs(body)

for rec in recommendations:
    print(f"Drug: {rec['drug_name']}")
    print(f"Mechanism: {rec['mechanism']}")
    print(f"Safety: {rec['safety_score']:.2f}")
```

---

## Running Examples

```bash
# Disease scenarios
python examples/demo_scenarios.py

# Drug discovery
python examples/demo_drug_discovery.py

# Geriatric medicine
python examples/demo_geriatric_medicine.py

# Diabetic ulcer treatment
python examples/demo_diabetic_ulcer_discovery.py

# Geroprotector trials
python examples/trial_geroprotectors.py
```

---

## Next Steps

1. Explore `examples/` directory for more demonstrations
2. Read `API_REFERENCE.md` for complete API documentation
3. See `RESEARCH_APPLICATIONS.md` for research use cases
4. Review component docs in `docs/` directory

---

**Happy researching!** 🔬
