# Deterministic Thinking Machine

> **A compiler for physical reality where laws are enforced, not simulated.**

[![License](https://img.shields.io/badge/license-MIT-blue.svg)](LICENSE)
[![Status](https://img.shields.io/badge/status-alpha-orange.svg)]()

---

## What This Is

A **physics law enforcement engine** combined with a **deterministic execution kernel**.

Not a simulator. Not an LLM. Not AGI.

A system that decides whether your physics is **legally valid** before allowing execution.

```text
Input: Physical system definition
Process: Law validation + constraint checking
Output: VALID | INVALID | INCOMPLETE
```

Same inputs → **same output forever**.

---

## The Core Insight

> **Most AI systems try to be smart. We try to be correct.**

Traditional approach:
```
Question → Calculation → Answer
```

Our approach:
```
Claim → Law Validation → ACCEPT | REJECT | INCOMPLETE
```

**Key difference**: We refuse to answer if the setup violates physics.

---

## What You Get

### ✅ Physics Verdict Objects

For any physics problem, you get:

```json
{
  "claim": "VALID",
  "laws_checked": ["conservation_of_energy", "newton_second_law"],
  "violations": [],
  "assumptions": ["closed_system", "non_relativistic"],
  "confidence": 0.97,
  "determinism": "PROVEN",
  "trace_hash": "0xPHYS91A"
}
```

### ✅ Formal Reasoning Traces

Not prose explanations. Machine-verifiable, human-auditable traces:

```
Given: System definition + Initial conditions + Constraints
Apply: Law selection + Invariant validation + Constraint propagation
Result: ACCEPT | REJECT | REFINE
```

### ✅ Pre-Simulation Validation

Before running expensive simulations:
1. Check if the setup is physically possible
2. Validate boundary conditions
3. Ensure conservation laws aren't violated
4. Verify constraints are compatible

**Save compute. Save time. Save embarrassment.**

---

## Architecture

### Two-Layer System

```text
┌─────────────────────────────────────┐
│   Physics Law Enforcement           │
│   (riemann-mandate)                 │
│   • Validates physical claims       │
│   • Enforces conservation laws      │
│   • Produces deterministic verdicts │
└─────────────┬───────────────────────┘
              │ Law ABI (strict contract)
              ▼
┌─────────────────────────────────────┐
│   Execution Kernel                  │
│   (bestbrain-core)                  │
│   • WASM-based execution            │
│   • ISL-safe routing                │
│   • Zero ambiguity                  │
└─────────────────────────────────────┘
```

### Key Properties

| Property | Guarantee |
|----------|-----------|
| **Determinism** | Same input → same output forever |
| **Law Enforcement** | Conservation laws are invariants |
| **Auditability** | Full reasoning traces |
| **Honesty** | Refuses incomplete setups |
| **Security** | No hallucination, no injection |

---

## Use Cases

### 🤖 Robotics
**Problem**: Motion planners propose physically impossible trajectories  
**Solution**: Reject them before execution

### 🔧 Engineering
**Problem**: Simulations run with invalid boundary conditions  
**Solution**: Validate constraints before simulation

### 📚 Education
**Problem**: Students memorize equations without understanding laws  
**Solution**: Teach why equations are allowed, not just how to solve them

### 🔬 Science
**Problem**: Papers make claims without validating assumptions  
**Solution**: Explicit, auditable physics reasoning traces

---

## Quick Example: Pendulum Validation

### Input

```json
{
  "system": {
    "type": "simple_pendulum",
    "parameters": {
      "length": 1.0,
      "mass": 0.5,
      "initial_angle": 0.1
    },
    "constraints": ["rigid_rod", "no_friction", "small_angle"]
  }
}
```

### Output (Physics Verdict)

```json
{
  "claim": "VALID",
  "laws_checked": [
    "conservation_of_energy",
    "newton_second_law"
  ],
  "violations": [],
  "assumptions": [
    {
      "description": "small_angle_approximation",
      "valid_when": "θ < 0.2 rad",
      "impact": "DEGRADES"
    }
  ],
  "confidence": 0.98,
  "determinism": "PROVEN"
}
```

### Kernel Decision

```json
{
  "decision": "ACCEPT",
  "confidence": 0.98,
  "reason": "Physics verdict VALID, all constraints satisfied",
  "isl_status": "SAFE"
}
```

---

## Repository Structure

```text
deterministic-thinking-machine/
├── kernel/
│   └── bestbrain-core/          # Execution kernel (submodule)
├── cognition/
│   └── riemann-mandate/         # Physics law enforcement (submodule)
├── interface/
│   ├── blackboard/              # Evidence store
│   ├── router/                  # Expert routing
│   └── evaluator/               # Confidence & ISL checking
├── examples/
│   ├── robotics/                # Motion planning validation
│   ├── security/                # Exploit detection
│   ├── reasoning/               # General physics problems
│   └── economics/               # Trading decision validation
├── tests/
│   ├── deterministic_tests/     # Same input → same output
│   ├── adversarial_tests/       # Malformed input handling
│   └── regression/              # No behavior changes
└── docs/
    ├── ARCHITECTURE.md          # System design
    ├── MANIFESTO.md             # Philosophy
    ├── LAW_ABI.md               # Interface specification
    ├── ISL_THEORY.md            # Inverse Scaling Law
    └── SCIENTIFIC_POSITION.md   # Academic justification
```

---

## What This System IS NOT

❌ **Not a simulator** - We validate, not simulate  
❌ **Not a neural network** - We enforce, not learn  
❌ **Not a probabilistic model** - We prove, not estimate  
❌ **Not AGI** - We know our limits  
❌ **Not an LLM wrapper** - No prompts, no hallucination

---

## Design Principles

### 1. Laws Outrank Answers
Conservation laws, Newton's laws, constraint equations - these are **invariants**, not suggestions.

### 2. Determinism First
Every computation must be reproducible.

### 3. Explicit Assumptions
If you assume a closed system, say so. If you can't state your assumptions, we reject your claim.

### 4. Silence Over Hallucination
Better to say "INCOMPLETE" than to guess.

### 5. Verification Over Capability
We'd rather do less and prove it, than do more and hope.

---

## Comparison to Other Systems

| System | Determinism | Law Enforcement | Hallucination | Auditability |
|--------|-------------|-----------------|---------------|--------------|
| **This System** | ✅ Guaranteed | ✅ Strict | ❌ Impossible | ✅ Full trace |
| Physics LLMs | ❌ Stochastic | ❌ None | ✅ Common | ❌ Opaque |
| Symbolic Solvers | ✅ Yes | ⚠️ Partial | ❌ No | ⚠️ Limited |
| Simulators | ⚠️ Numeric | ❌ None | ⚠️ Drift | ⚠️ Logs only |

---

## Getting Started

### Prerequisites

- Rust (for WASM compilation)
- Node.js (for TypeScript interface)
- Docker (optional, for containerized deployment)

### Installation

```bash
# Clone the repository
git clone <repo-url> deterministic-thinking-machine
cd deterministic-thinking-machine

# Initialize submodules
git submodule update --init --recursive

# Build the kernel
cd kernel/bestbrain-core
cargo build --release --target wasm32-unknown-unknown

# Build the physics layer
cd ../../cognition/riemann-mandate
npm install
npm run build

# Run tests
npm test
```

### Run an Example

```bash
# Validate a pendulum system
cd examples/robotics
node validate_pendulum.js

# Expected output:
# ✅ Physics verdict: VALID
# ✅ Kernel decision: ACCEPT
# ✅ Confidence: 0.98
# ✅ Determinism: PROVEN
```

---

## Documentation

- **[ARCHITECTURE.md](docs/ARCHITECTURE.md)** - System design and layer responsibilities
- **[MANIFESTO.md](docs/MANIFESTO.md)** - Philosophy and core principles
- **[LAW_ABI.md](docs/LAW_ABI.md)** - Interface specification between layers
- **[ISL_THEORY.md](docs/ISL_THEORY.md)** - Inverse Scaling Law rationale
- **[SCIENTIFIC_POSITION.md](docs/SCIENTIFIC_POSITION.md)** - Academic justification

---

## Who This Is For

### ✅ Engineers
Who need to validate systems before building them.

### ✅ Roboticists
Who need provably safe motion planning.

### ✅ Scientists
Who need auditable physics reasoning.

### ✅ Educators
Who want to teach thinking, not memorization.

### ✅ Skeptics
Who don't trust AI claims without proof.

---

## Contributing

We welcome contributions that maintain the core principles:

1. **Determinism** - No randomness
2. **Law enforcement** - Physics laws are invariants
3. **Auditability** - Full reasoning traces
4. **Honesty** - Refuse incomplete setups

See [CONTRIBUTING.md](CONTRIBUTING.md) for guidelines.

---

## License

MIT License - see [LICENSE](LICENSE) for details.

---

## Citation

If you use this system in academic work, please cite:

```bibtex
@software{deterministic_thinking_machine,
  title = {Deterministic Thinking Machine: A Compiler for Physical Reality},
  author = {[Your Name]},
  year = {2025},
  url = {[repo-url]}
}
```

---

## Contact

- **Issues**: [GitHub Issues](issues)
- **Discussions**: [GitHub Discussions](discussions)
- **Email**: [your-email]

---

## Acknowledgments

This system builds on:
- **bestbrain-core**: Deterministic execution kernel
- **riemann-mandate**: Physics law enforcement engine

---

## Status

**Alpha** - Core functionality complete, API may change.

### Roadmap

- [x] Core architecture
- [x] Law ABI specification
- [x] Basic physics validators
- [ ] Complete law registry
- [ ] Robotics examples
- [ ] Security examples
- [ ] Academic paper
- [ ] Production deployment guide

---

## Final Statement

We are not building AGI.  
We are not building a chatbot.  
We are not building a simulator.

**We are building a compiler for physical reality.**

If your physics doesn't compile, we won't run it.

That's the promise.
