# API Reference
## Complete API Documentation

---

## Body Model API

### HumanBodyState

Main class representing complete human physiology.

```python
from body_model.body_state_model import HumanBodyState

body = HumanBodyState()
```

#### Organ Access

```python
# Access individual organs
body.heart       # HeartState
body.lungs       # LungState
body.liver       # LiverState
body.kidneys     # KidneyState
body.brain       # BrainState
body.pancreas    # PancreasState
body.gut         # GutState
body.immune      # ImmuneState
body.endocrine   # EndocrineState
body.muscles     # MuscleState
body.bones       # BoneState
body.skin        # SkinState
```

#### Methods

**get_all_organs()**
```python
organs = body.get_all_organs()
# Returns: dict of {organ_name: organ_instance}
```

**reset_to_basal()**
```python
body.reset_to_basal()
# Resets all organs to healthy baseline
```

**get_overall_homeostasis()**
```python
homeostasis = body.get_overall_homeostasis()
# Returns: {
#   "overall_homeostasis_score": float,
#   "organ_scores": dict,
#   "imbalanced_organs": list
# }
```

**get_state_summary()**
```python
summary = body.get_state_summary()
# Returns: str (human-readable summary)
```

---

### BodySystemOrchestrator

Manages inter-organ communication and simulation.

```python
from body_model.body_state_model import BodySystemOrchestrator

orchestrator = BodySystemOrchestrator(body)
```

#### Methods

**run_simulation(duration_seconds, verbose=False)**
```python
orchestrator.run_simulation(
    duration_seconds=60.0,
    verbose=True
)
# Simulates physiological state for specified duration
```

**step()**
```python
orchestrator.step()
# Execute one simulation time step
```

**get_history_summary()**
```python
history = orchestrator.get_history_summary()
# Returns: dict with time series data
```

---

## Organ State APIs

### HeartState

```python
# State variables
body.heart.heart_rate_bpm           # Heart rate (beats/min)
body.heart.stroke_volume_ml         # Stroke volume (mL)
body.heart.cardiac_output_l_min     # Cardiac output (L/min)
body.heart.ejection_fraction        # Ejection fraction (0-1)
body.heart.systolic_bp_mmhg         # Systolic BP (mmHg)
body.heart.diastolic_bp_mmhg        # Diastolic BP (mmHg)

# Disease transitions
body.heart.transition_to_heart_failure(cause="ischemic")
body.heart.transition_to_hypertrophy(cause="hypertension")
body.heart.transition_to_arrhythmia(type="atrial_fibrillation")
```

### KidneyState

```python
# State variables
body.kidneys.gfr_ml_min             # GFR (mL/min)
body.kidneys.creatinine_mg_dl       # Creatinine (mg/dL)
body.kidneys.bun_mg_dl              # BUN (mg/dL)
body.kidneys.urine_output_ml_day    # Urine output (mL/day)

# Disease transitions
body.kidneys.transition_to_ckd(stage=3)  # CKD stage 1-5
body.kidneys.transition_to_aki(severity="moderate")
```

### BrainState

```python
# State variables
body.brain.cerebral_blood_flow_ml_min  # CBF (mL/min)
body.brain.consciousness_level         # 0-100
body.brain.cognitive_score             # 0-100

# Disease transitions
body.brain.transition_to_alzheimers(stage="mild")
body.brain.transition_to_parkinsons(stage="early")
body.brain.transition_to_stroke(type="ischemic")
```

### Other Organs

Similar APIs for:
- `LungState`, `LiverState`, `PancreasState`
- `GutState`, `ImmuneState`, `EndocrineState`
- `MuscleState`, `BoneState`, `SkinState`

See `docs/README_BODY_MODEL.md` for complete organ documentation.

---

## NanoCERN Reactor API

### MinimalReactor

```python
from nanocern_reactor.core import MinimalReactor

reactor = MinimalReactor()
```

**execute_reaction(state, kus)**
```python
result = reactor.execute_reaction(
    state={"GFR": 25, "age": 75},
    kus=medical_kus
)

# Returns: {
#   "accepted_kus": list,
#   "nkus": list,  # Violations
#   "timestamp": float
# }
```

### Medical Knowledge Units

```python
from nanocern_reactor.flagship_medical_kus import get_flagship_medical_kus

medical_kus = get_flagship_medical_kus()
# Returns: list of medical safety KUs
```

### HomeostasisEngine

```python
from nanocern_reactor.homeostasis_engine import HomeostasisEngine

engine = HomeostasisEngine()

verdict = engine.validate_intervention(
    patient_state=body,
    drug="metformin",
    dose=1000
)
```

---

## Drug Discovery API

### DrugDiscoveryEngine

```python
from drug_discovery.discovery_engine import DrugDiscoveryEngine

engine = DrugDiscoveryEngine()
```

**discover_drugs(body_state)**
```python
recommendations = engine.discover_drugs(body)

# Returns: list of {
#   "drug_name": str,
#   "mechanism": str,
#   "target_organs": list,
#   "safety_score": float,
#   "contraindications": list
# }
```

### DeprescribingEngine

```python
from drug_discovery.deprescribing_engine import DeprescribingEngine

engine = DeprescribingEngine()

recommendations = engine.analyze_medications(
    age=85,
    medications=["benzodiazepine", "anticholinergic"],
    kidney_function=45
)
```

### UlcerDrugDiscoveryEngine

```python
from drug_discovery.ulcer_discovery_engine import UlcerDrugDiscoveryEngine

engine = UlcerDrugDiscoveryEngine()

protocol = engine.design_treatment_protocol(
    ulcer_grade=3,
    infection_present=True,
    patient_age=65
)
```

---

## Longevity API

### LongevityOrchestrator

```python
from body_model.longevity_orchestrator import LongevityOrchestrator

longevity = LongevityOrchestrator(body)
```

**simulate_lifespan(years, genetic_profile, verbose=True)**
```python
history = longevity.simulate_lifespan(
    years=100,
    genetic_profile=["APOE_e2", "FOXO3_longevity"],
    verbose=True
)

# Returns: list of {
#   "age": int,
#   "homeostasis": float
# }
```

### GenomicsEngine

```python
from body_model.genetics_module import GenomicsEngine

genetics = GenomicsEngine()

multiplier = genetics.calculate_integrated_multiplier(
    genetic_profile=["APOE_e2"],
    organ="heart"
)
# Returns: float (senescence rate multiplier)
```

---

## Common Patterns

### Pattern 1: Disease Cascade Analysis

```python
# Induce primary disease
body.heart.transition_to_heart_failure("ischemic")

# Run simulation
orchestrator.run_simulation(3600.0)  # 1 hour

# Analyze cascade effects
for organ_name, organ in body.get_all_organs().items():
    score = organ.calculate_homeostasis()
    print(f"{organ_name}: {score:.1f}%")
```

### Pattern 2: Drug Safety Validation

```python
# Check contraindications
patient_state = {
    "GFR": body.kidneys.gfr_ml_min,
    "age": 75,
    "liver_function": body.liver.albumin_g_dl
}

result = reactor.execute_reaction(patient_state, medical_kus)

if result['nkus']:
    print("⚠️ Contraindications detected!")
    for nku in result['nkus']:
        print(f"  - {nku['reason']}")
```

### Pattern 3: Longitudinal Study

```python
# Track disease progression over time
history = []

for day in range(365):
    orchestrator.run_simulation(86400.0)  # 24 hours
    homeostasis = body.get_overall_homeostasis()
    history.append({
        "day": day,
        "homeostasis": homeostasis['overall_homeostasis_score']
    })
```

---

## Error Handling

```python
try:
    body.heart.transition_to_heart_failure("ischemic")
    orchestrator.run_simulation(60.0)
except Exception as e:
    print(f"Simulation error: {e}")
```

---

## Performance Tips

1. **Batch simulations**: Run longer durations instead of many short ones
2. **Disable verbose**: Set `verbose=False` for faster execution
3. **Reset state**: Use `body.reset_to_basal()` between experiments
4. **Profile code**: Use Python profiler for optimization

---

See individual component documentation for more details.
