"""
Skin (Integumentary) State Machine.

Models skin physiology including:
- Barrier function
- Thermoregulation
- Immune defense
- Disease states (dermatitis, psoriasis, aging)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class SkinState(OrganState):
    """Skin state machine."""
    
    organ_name: str = "Skin"
    
    # Structural
    surface_area_m2: float = 2.0
    thickness_mm: float = 2.0
    collagen_content_percent: float = 100.0
    elastin_content_percent: float = 100.0
    
    # Cascade Triggers
    peripheral_edema_index: float = 0.0  # 0.0 = none
    
    # Barrier function
    barrier_integrity_percent: float = 100.0
    transepidermal_water_loss_g_m2_h: float = 10.0
    ph: float = 5.5
    lipid_content_percent: float = 100.0
    
    # Immune function
    langerhans_cell_density_percent: float = 100.0
    antimicrobial_peptide_production_percent: float = 100.0
    inflammatory_response_percent: float = 0.0
    
    # Thermoregulation
    sweat_production_ml_hr: float = 50.0
    blood_flow_ml_min_100g: float = 5.0
    temperature_regulation_percent: float = 100.0
    
    # Regeneration
    cell_turnover_days: float = 28.0
    wound_healing_rate_percent: float = 100.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal skin state."""
        return {
            "surface_area_m2": 2.0,
            "thickness_mm": 2.0,
            "collagen_content_percent": 100.0,
            "elastin_content_percent": 100.0,
            "peripheral_edema_index": 0.0,
            "barrier_integrity_percent": 100.0,
            "transepidermal_water_loss_g_m2_h": 10.0,
            "ph": 5.5,
            "lipid_content_percent": 100.0,
            "langerhans_cell_density_percent": 100.0,
            "antimicrobial_peptide_production_percent": 100.0,
            "inflammatory_response_percent": 0.0,
            "sweat_production_ml_hr": 50.0,
            "blood_flow_ml_min_100g": 5.0,
            "temperature_regulation_percent": 100.0,
            "cell_turnover_days": 28.0,
            "wound_healing_rate_percent": 100.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all skin state variables."""
        return [
            "surface_area_m2", "thickness_mm", "collagen_content_percent",
            "elastin_content_percent", "peripheral_edema_index", "barrier_integrity_percent",
            "transepidermal_water_loss_g_m2_h", "ph", "lipid_content_percent",
            "langerhans_cell_density_percent",
            "antimicrobial_peptide_production_percent",
            "inflammatory_response_percent", "sweat_production_ml_hr",
            "blood_flow_ml_min_100g", "temperature_regulation_percent",
            "cell_turnover_days", "wound_healing_rate_percent"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_dermatitis(self, severity: str = "moderate"):
        """Transition to atopic dermatitis/eczema."""
        severity_map = {"mild": 0.7, "moderate": 0.5, "severe": 0.3}
        function_factor = severity_map.get(severity, 0.5)
        
        # Barrier dysfunction
        self.barrier_integrity_percent = 100.0 * function_factor
        self.transepidermal_water_loss_g_m2_h = 10.0 + ((1 - function_factor) * 40.0)
        self.lipid_content_percent = 100.0 * function_factor
        
        # Inflammation
        self.inflammatory_response_percent = (1 - function_factor) * 100.0
        
        self.state_classification = "INFLAMED"
    
    def transition_to_psoriasis(self):
        """Transition to psoriasis."""
        # Hyperproliferation
        self.cell_turnover_days = 4.0  # ↓86%
        
        # Inflammation
        self.inflammatory_response_percent = 80.0
        
        # Barrier changes
        self.barrier_integrity_percent = 60.0
        
        self.state_classification = "PSORIATIC"
    
    def transition_to_aging(self, age_years: float = 70.0):
        """Transition to aged skin."""
        aging_factor = (age_years - 30) / 70.0  # Normalized to 30-100 years
        
        # Structural changes
        self.collagen_content_percent = 100.0 * (1 - aging_factor * 0.5)
        self.elastin_content_percent = 100.0 * (1 - aging_factor * 0.6)
        self.thickness_mm = 2.0 * (1 - aging_factor * 0.3)
        
        # Functional decline
        self.wound_healing_rate_percent = 100.0 * (1 - aging_factor * 0.4)
        self.barrier_integrity_percent = 100.0 * (1 - aging_factor * 0.3)
        
        self.state_classification = "AGED"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Impaired thermoregulation affects core temperature
        if self.temperature_regulation_percent < 70:
            effects["Core"] = {
                "temperature_dysregulation": (100 - self.temperature_regulation_percent) / 100.0
            }
        
        return effects
