"""
Pancreas State Machine.

Models complete pancreatic physiology including:
- Endocrine function (insulin, glucagon from islets)
- Exocrine function (digestive enzymes)
- Disease states (Type 1/2 diabetes, pancreatitis)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class PancreasState(OrganState):
    """Pancreatic state machine."""
    
    organ_name: str = "Pancreas"
    
    # Structural parameters
    mass_g: float = 100.0
    beta_cell_mass_mg: float = 1000.0
    alpha_cell_mass_mg: float = 300.0
    acinar_cell_mass_g: float = 80.0
    
    # Endocrine function (Islets of Langerhans)
    insulin_secretion_mu_ml: float = 10.0
    glucagon_secretion_pg_ml: float = 50.0
    insulin_sensitivity_index: float = 100.0
    c_peptide_ng_ml: float = 2.0  # Marker of insulin production
    
    # Exocrine function
    enzyme_secretion_ml_day: float = 1500.0
    amylase_u_l: float = 70.0
    lipase_u_l: float = 30.0
    trypsin_activity_percent: float = 100.0
    digestive_capacity_percent: float = 100.0
    
    # Biomarkers
    hba1c_percent: float = 5.0  # Glycated hemoglobin
    fasting_glucose_mg_dl: float = 90.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal pancreatic state."""
        return {
            "mass_g": 100.0,
            "beta_cell_mass_mg": 1000.0,
            "alpha_cell_mass_mg": 300.0,
            "acinar_cell_mass_g": 80.0,
            "insulin_secretion_mu_ml": 10.0,
            "glucagon_secretion_pg_ml": 50.0,
            "insulin_sensitivity_index": 100.0,
            "c_peptide_ng_ml": 2.0,
            "enzyme_secretion_ml_day": 1500.0,
            "amylase_u_l": 70.0,
            "lipase_u_l": 30.0,
            "trypsin_activity_percent": 100.0,
            "digestive_capacity_percent": 100.0,
            "hba1c_percent": 5.0,
            "fasting_glucose_mg_dl": 90.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all pancreatic state variables."""
        return [
            "mass_g", "beta_cell_mass_mg", "alpha_cell_mass_mg", "acinar_cell_mass_g",
            "insulin_secretion_mu_ml", "glucagon_secretion_pg_ml",
            "insulin_sensitivity_index", "c_peptide_ng_ml",
            "enzyme_secretion_ml_day", "amylase_u_l", "lipase_u_l",
            "trypsin_activity_percent", "digestive_capacity_percent",
            "hba1c_percent", "fasting_glucose_mg_dl"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_type1_diabetes(self, autoimmune_destruction_percent: float = 90.0):
        """
        Transition to Type 1 diabetes (autoimmune beta cell destruction).
        
        Args:
            autoimmune_destruction_percent: Percentage of beta cells destroyed
        """
        remaining_beta_cells = (100 - autoimmune_destruction_percent) / 100.0
        
        # Beta cell loss
        self.beta_cell_mass_mg = 1000.0 * remaining_beta_cells
        
        # Insulin deficiency
        self.insulin_secretion_mu_ml = 10.0 * remaining_beta_cells
        self.c_peptide_ng_ml = 2.0 * remaining_beta_cells
        
        # Compensatory glucagon excess
        self.glucagon_secretion_pg_ml = 50.0 + ((1 - remaining_beta_cells) * 100.0)
        
        # Hyperglycemia
        self.fasting_glucose_mg_dl = 90.0 + ((1 - remaining_beta_cells) * 210.0)
        self.hba1c_percent = 5.0 + ((1 - remaining_beta_cells) * 7.0)
        
        self.state_classification = "DIABETIC"
    
    def transition_to_type2_diabetes(self, insulin_resistance_percent: float = 70.0):
        """
        Transition to Type 2 diabetes (insulin resistance + beta cell dysfunction).
        
        Args:
            insulin_resistance_percent: Degree of insulin resistance (0-100%)
        """
        resistance_factor = insulin_resistance_percent / 100.0
        
        # Insulin resistance
        self.insulin_sensitivity_index = 100.0 * (1 - resistance_factor)
        
        # Compensatory hyperinsulinemia (early stage)
        if resistance_factor < 0.7:
            self.insulin_secretion_mu_ml = 10.0 * (1 + resistance_factor * 2.0)
        else:
            # Beta cell exhaustion (late stage)
            beta_cell_failure = (resistance_factor - 0.7) / 0.3
            self.beta_cell_mass_mg = 1000.0 * (1 - beta_cell_failure * 0.5)
            self.insulin_secretion_mu_ml = 10.0 * (1 - beta_cell_failure * 0.5)
        
        # Hyperglycemia
        self.fasting_glucose_mg_dl = 90.0 + (resistance_factor * 130.0)
        self.hba1c_percent = 5.0 + (resistance_factor * 5.0)
        
        self.state_classification = "DIABETIC"
    
    def transition_to_pancreatitis(self, severity: str = "moderate"):
        """
        Transition to acute pancreatitis.
        
        Args:
            severity: "mild", "moderate", "severe"
        """
        severity_map = {
            "mild": 0.7,
            "moderate": 0.5,
            "severe": 0.2
        }
        
        function_factor = severity_map.get(severity, 0.5)
        
        # Exocrine dysfunction
        self.enzyme_secretion_ml_day = 1500.0 * function_factor
        self.digestive_capacity_percent = 100.0 * function_factor
        
        # Enzyme leakage into blood
        self.amylase_u_l = 70.0 + ((1 - function_factor) * 930.0)  # Can reach 1000+
        self.lipase_u_l = 30.0 + ((1 - function_factor) * 970.0)  # Can reach 1000+
        
        # Endocrine impairment (if severe)
        if severity == "severe":
            self.insulin_secretion_mu_ml = 10.0 * function_factor
            self.fasting_glucose_mg_dl = 90.0 + ((1 - function_factor) * 110.0)
        
        self.state_classification = "STRESSED"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Insulin affects liver glucose production
        effects["Liver"] = {
            "glucose_production_mg_dl_hr": -100.0 * (self.insulin_secretion_mu_ml / 10.0 - 1)
        }
        
        # Insulin affects muscle glucose uptake
        effects["Muscles"] = {
            "glucose_uptake_percent": 50.0 * (self.insulin_secretion_mu_ml / 10.0)
        }
        
        # Reduced enzymes affect gut digestion
        if self.digestive_capacity_percent < 70:
            effects["Gut"] = {
                "nutrient_absorption_percent": -(100 - self.digestive_capacity_percent) / 2.0
            }
        
        return effects
    
    def receive_from_gut(self, blood_glucose_mg_dl: float):
        """
        Receive glucose signal from gut/blood.
        
        Args:
            blood_glucose_mg_dl: Blood glucose level
        """
        self.fasting_glucose_mg_dl = blood_glucose_mg_dl
        
        # Glucose-stimulated insulin secretion
        if blood_glucose_mg_dl > 100:
            glucose_stimulus = (blood_glucose_mg_dl - 90) / 100.0
            self.insulin_secretion_mu_ml = 10.0 + (glucose_stimulus * 20.0)
            self.glucagon_secretion_pg_ml = 50.0 - (glucose_stimulus * 30.0)
        elif blood_glucose_mg_dl < 80:
            hypoglycemia = (90 - blood_glucose_mg_dl) / 90.0
            self.insulin_secretion_mu_ml = 10.0 * (1 - hypoglycemia)
            self.glucagon_secretion_pg_ml = 50.0 + (hypoglycemia * 100.0)
