"""
Muscle (Musculoskeletal) State Machine.

Models skeletal muscle physiology including:
- Strength and mass
- Metabolism and energy
- Protein synthesis
- Disease states (sarcopenia, myopathy, cachexia)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class MuscleState(OrganState):
    """Muscle state machine."""
    
    organ_name: str = "Muscles"
    
    # Structural
    muscle_mass_kg: float = 30.0
    muscle_fiber_count_billion: float = 5.0
    muscle_fiber_diameter_um: float = 50.0
    
    # Strength and function
    max_strength_percent: float = 100.0
    endurance_percent: float = 100.0
    power_output_watts: float = 300.0
    
    # Metabolism
    glucose_uptake_percent: float = 100.0
    glycogen_storage_g: float = 400.0
    fat_oxidation_percent: float = 100.0
    metabolic_rate_percent: float = 100.0
    
    # Protein metabolism
    protein_synthesis_percent: float = 100.0
    protein_degradation_percent: float = 100.0
    myostatin_ng_ml: float = 5.0  # Muscle growth inhibitor
    
    # Energy and fatigue
    atp_stores_percent: float = 100.0
    creatine_phosphate_percent: float = 100.0
    lactate_threshold_percent: float = 100.0
    fatigue_resistance_percent: float = 100.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal muscle state."""
        return {
            "muscle_mass_kg": 30.0,
            "muscle_fiber_count_billion": 5.0,
            "muscle_fiber_diameter_um": 50.0,
            "max_strength_percent": 100.0,
            "endurance_percent": 100.0,
            "power_output_watts": 300.0,
            "glucose_uptake_percent": 100.0,
            "glycogen_storage_g": 400.0,
            "fat_oxidation_percent": 100.0,
            "metabolic_rate_percent": 100.0,
            "protein_synthesis_percent": 100.0,
            "protein_degradation_percent": 100.0,
            "myostatin_ng_ml": 5.0,
            "atp_stores_percent": 100.0,
            "creatine_phosphate_percent": 100.0,
            "lactate_threshold_percent": 100.0,
            "fatigue_resistance_percent": 100.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all muscle state variables."""
        return [
            "muscle_mass_kg", "muscle_fiber_count_billion", "muscle_fiber_diameter_um",
            "max_strength_percent", "endurance_percent", "power_output_watts",
            "glucose_uptake_percent", "glycogen_storage_g", "fat_oxidation_percent",
            "metabolic_rate_percent", "protein_synthesis_percent",
            "protein_degradation_percent", "myostatin_ng_ml",
            "atp_stores_percent", "creatine_phosphate_percent",
            "lactate_threshold_percent", "fatigue_resistance_percent"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_sarcopenia(self, age_related_loss_percent: float = 30.0):
        """Transition to sarcopenia (age-related muscle loss)."""
        loss_factor = age_related_loss_percent / 100.0
        
        self.muscle_mass_kg = 30.0 * (1 - loss_factor)
        self.muscle_fiber_count_billion = 5.0 * (1 - loss_factor)
        self.max_strength_percent = 100.0 * (1 - loss_factor)
        self.power_output_watts = 300.0 * (1 - loss_factor)
        
        # Metabolic changes
        self.protein_synthesis_percent = 100.0 * (1 - loss_factor * 0.6)
        self.protein_degradation_percent = 100.0 * (1 + loss_factor * 0.4)
        
        self.state_classification = "SARCOPENIC"
    
    def transition_to_cachexia(self):
        """Transition to cachexia (wasting syndrome)."""
        self.muscle_mass_kg = 18.0  # ↓40%
        self.max_strength_percent = 50.0
        
        # Hypermetabolic state
        self.protein_degradation_percent = 200.0
        self.protein_synthesis_percent = 40.0
        
        self.state_classification = "CACHECTIC"
    
    def transition_to_myopathy(self, severity: str = "moderate"):
        """Transition to myopathy (muscle disease)."""
        severity_map = {"mild": 0.7, "moderate": 0.5, "severe": 0.3}
        function_factor = severity_map.get(severity, 0.5)
        
        self.max_strength_percent = 100.0 * function_factor
        self.endurance_percent = 100.0 * function_factor
        self.fatigue_resistance_percent = 100.0 * function_factor
        
        self.state_classification = "MYOPATHIC"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Muscle mass affects metabolism
        if self.muscle_mass_kg < 20:
            effects["Pancreas"] = {
                "insulin_sensitivity_index": -(30 - self.muscle_mass_kg) * 2.0
            }
        
        return effects
    
    def receive_from_pancreas(self, insulin_level: float):
        """Receive insulin signal."""
        self.glucose_uptake_percent = 100.0 * (insulin_level / 10.0)
