"""
Lung (Pulmonary) State Machine.

Models complete pulmonary physiology including:
- Ventilation mechanics
- Gas exchange
- Perfusion
- Disease states (COPD, asthma, pneumonia)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class LungState(OrganState):
    """Pulmonary state machine."""
    
    organ_name: str = "Lungs"
    
    # Structural parameters
    total_lung_capacity_l: float = 6.0
    vital_capacity_l: float = 4.8
    residual_volume_l: float = 1.2
    alveolar_surface_area_m2: float = 70.0
    
    # Ventilation parameters
    respiratory_rate_per_min: float = 12.0
    tidal_volume_ml: float = 500.0
    minute_ventilation_l_min: float = 6.0
    dead_space_ml: float = 150.0
    fev1_percent_predicted: float = 100.0  # Forced expiratory volume
    
    # Gas exchange parameters
    arterial_po2_mmhg: float = 95.0
    arterial_pco2_mmhg: float = 40.0
    arterial_oxygen_saturation_percent: float = 98.0
    alveolar_arterial_gradient_mmhg: float = 10.0
    
    # Mechanics parameters
    lung_compliance_ml_cmh2o: float = 200.0
    airway_resistance_cmh2o_l_s: float = 2.0
    work_of_breathing_percent: float = 100.0
    
    # Perfusion parameters
    pulmonary_blood_flow_l_min: float = 5.0
    ventilation_perfusion_ratio: float = 0.8
    shunt_fraction_percent: float = 2.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal pulmonary state."""
        return {
            "total_lung_capacity_l": 6.0,
            "vital_capacity_l": 4.8,
            "residual_volume_l": 1.2,
            "alveolar_surface_area_m2": 70.0,
            "respiratory_rate_per_min": 12.0,
            "tidal_volume_ml": 500.0,
            "minute_ventilation_l_min": 6.0,
            "dead_space_ml": 150.0,
            "fev1_percent_predicted": 100.0,
            "arterial_po2_mmhg": 95.0,
            "arterial_pco2_mmhg": 40.0,
            "arterial_oxygen_saturation_percent": 98.0,
            "alveolar_arterial_gradient_mmhg": 10.0,
            "lung_compliance_ml_cmh2o": 200.0,
            "airway_resistance_cmh2o_l_s": 2.0,
            "work_of_breathing_percent": 100.0,
            "pulmonary_blood_flow_l_min": 5.0,
            "ventilation_perfusion_ratio": 0.8,
            "shunt_fraction_percent": 2.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all pulmonary state variables."""
        return [
            "total_lung_capacity_l", "vital_capacity_l", "residual_volume_l",
            "alveolar_surface_area_m2", "respiratory_rate_per_min", "tidal_volume_ml",
            "minute_ventilation_l_min", "dead_space_ml", "fev1_percent_predicted",
            "arterial_po2_mmhg", "arterial_pco2_mmhg", "arterial_oxygen_saturation_percent",
            "alveolar_arterial_gradient_mmhg", "lung_compliance_ml_cmh2o",
            "airway_resistance_cmh2o_l_s", "work_of_breathing_percent",
            "pulmonary_blood_flow_l_min", "ventilation_perfusion_ratio",
            "shunt_fraction_percent"
        ]
    
    def update_minute_ventilation(self):
        """Update minute ventilation based on RR and TV."""
        self.minute_ventilation_l_min = (self.respiratory_rate_per_min * self.tidal_volume_ml) / 1000.0
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_copd(self, severity: str = "moderate"):
        """
        Transition to chronic obstructive pulmonary disease.
        
        Args:
            severity: "mild", "moderate", "severe", "very_severe"
        """
        severity_map = {
            "mild": 0.7,
            "moderate": 0.5,
            "severe": 0.35,
            "very_severe": 0.25
        }
        
        fev1_factor = severity_map.get(severity, 0.5)
        
        # Airflow obstruction
        self.fev1_percent_predicted = fev1_factor * 100.0
        self.airway_resistance_cmh2o_l_s = 2.0 / fev1_factor  # Increased resistance
        
        # Gas exchange impairment
        self.arterial_po2_mmhg = 70.0 + (fev1_factor * 25.0)
        self.arterial_pco2_mmhg = 40.0 + ((1 - fev1_factor) * 20.0)
        self.arterial_oxygen_saturation_percent = 88.0 + (fev1_factor * 10.0)
        
        # Increased work of breathing
        self.work_of_breathing_percent = 100.0 + ((1 - fev1_factor) * 200.0)
        
        # Hyperinflation
        self.residual_volume_l = 1.2 + ((1 - fev1_factor) * 1.5)
        
        self.state_classification = "OBSTRUCTED"
        self.update_minute_ventilation()
    
    def transition_to_asthma_attack(self, severity: str = "moderate"):
        """
        Transition to acute asthma exacerbation.
        
        Args:
            severity: "mild", "moderate", "severe"
        """
        if severity == "mild":
            self.fev1_percent_predicted = 70.0
            self.respiratory_rate_per_min = 20.0
            self.arterial_po2_mmhg = 90.0
        elif severity == "moderate":
            self.fev1_percent_predicted = 50.0
            self.respiratory_rate_per_min = 28.0
            self.arterial_po2_mmhg = 80.0
            self.arterial_pco2_mmhg = 42.0
        else:  # severe
            self.fev1_percent_predicted = 30.0
            self.respiratory_rate_per_min = 35.0
            self.arterial_po2_mmhg = 70.0
            self.arterial_pco2_mmhg = 50.0
        
        # Bronchospasm
        self.airway_resistance_cmh2o_l_s = 8.0
        self.work_of_breathing_percent = 300.0
        
        self.state_classification = "STRESSED"
        self.update_minute_ventilation()
    
    def transition_to_pneumonia(self, affected_lobes: int = 1):
        """
        Transition to pneumonia.
        
        Args:
            affected_lobes: Number of lung lobes affected (1-5)
        """
        consolidation_fraction = affected_lobes / 5.0
        
        # Reduced gas exchange surface
        self.alveolar_surface_area_m2 = 70.0 * (1 - consolidation_fraction * 0.6)
        
        # Hypoxemia
        self.arterial_po2_mmhg = 95.0 - (consolidation_fraction * 30.0)
        self.arterial_oxygen_saturation_percent = 98.0 - (consolidation_fraction * 15.0)
        
        # Increased shunt
        self.shunt_fraction_percent = 2.0 + (consolidation_fraction * 20.0)
        
        # Compensatory tachypnea
        self.respiratory_rate_per_min = 12.0 + (consolidation_fraction * 20.0)
        
        self.state_classification = "STRESSED"
        self.update_minute_ventilation()
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Hypoxia affects heart
        if self.arterial_po2_mmhg < 80:
            hypoxia_severity = (80 - self.arterial_po2_mmhg) / 80.0
            effects["Heart"] = {
                "heart_rate_bpm": 20.0 * hypoxia_severity,
                "contractility_percent": 15.0 * hypoxia_severity
            }
        
        # Hypoxia affects brain
        if self.arterial_po2_mmhg < 60:
            effects["Brain"] = {
                "consciousness_impairment": (60 - self.arterial_po2_mmhg) / 60.0
            }
        
        # Hypercapnia affects brain
        if self.arterial_pco2_mmhg > 50:
            effects["Brain"] = effects.get("Brain", {})
            effects["Brain"]["cerebral_vasodilation"] = (self.arterial_pco2_mmhg - 40) / 40.0
        
        return effects
    
    def receive_from_heart(self, cardiac_output_l_min: float):
        """
        Receive blood flow from heart.
        
        Args:
            cardiac_output_l_min: Cardiac output
        """
        self.pulmonary_blood_flow_l_min = cardiac_output_l_min
        
        # Update V/Q ratio
        if self.minute_ventilation_l_min > 0:
            self.ventilation_perfusion_ratio = self.minute_ventilation_l_min / self.pulmonary_blood_flow_l_min
