"""
Liver (Hepatic) State Machine.

Models complete hepatic physiology including:
- Synthetic function (albumin, clotting factors)
- Metabolic function (glucose, lipids)
- Detoxification (cytochrome P450, phase I/II)
- Excretion (bile, bilirubin)
- Disease states (cirrhosis, hepatitis, fatty liver)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class LiverState(OrganState):
    """Hepatic state machine."""
    
    organ_name: str = "Liver"
    
    # Structural parameters
    mass_g: float = 1500.0
    hepatocyte_count_billion: float = 240.0
    sinusoidal_perfusion_percent: float = 100.0
    hepatic_blood_flow_ml_min: float = 1500.0
    
    # Synthetic function
    albumin_g_dl: float = 4.0
    prothrombin_time_sec: float = 12.0
    inr: float = 1.0  # International normalized ratio
    clotting_factor_production_percent: float = 100.0
    
    # Metabolic function
    glucose_production_mg_dl_hr: float = 200.0
    glycogen_storage_g: float = 100.0
    gluconeogenesis_rate_percent: float = 100.0
    lipid_metabolism_percent: float = 100.0
    cholesterol_synthesis_mg_day: float = 1000.0
    
    # Detoxification
    cytochrome_p450_activity_percent: float = 100.0
    phase_1_metabolism_percent: float = 100.0
    phase_2_conjugation_percent: float = 100.0
    glutathione_levels_mmol: float = 5.0
    
    # Excretion
    bile_production_ml_day: float = 600.0
    bilirubin_mg_dl: float = 0.8
    ammonia_clearance_percent: float = 100.0
    ammonia_blood_level_umol_l: float = 30.0
    
    # Biomarkers
    alt_u_l: float = 25.0  # Alanine aminotransferase
    ast_u_l: float = 30.0  # Aspartate aminotransferase
    alkaline_phosphatase_u_l: float = 70.0
    ggt_u_l: float = 30.0  # Gamma-glutamyl transferase
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal hepatic state."""
        return {
            "mass_g": 1500.0,
            "hepatocyte_count_billion": 240.0,
            "sinusoidal_perfusion_percent": 100.0,
            "hepatic_blood_flow_ml_min": 1500.0,
            "albumin_g_dl": 4.0,
            "prothrombin_time_sec": 12.0,
            "inr": 1.0,
            "clotting_factor_production_percent": 100.0,
            "glucose_production_mg_dl_hr": 200.0,
            "glycogen_storage_g": 100.0,
            "gluconeogenesis_rate_percent": 100.0,
            "lipid_metabolism_percent": 100.0,
            "cholesterol_synthesis_mg_day": 1000.0,
            "cytochrome_p450_activity_percent": 100.0,
            "phase_1_metabolism_percent": 100.0,
            "phase_2_conjugation_percent": 100.0,
            "glutathione_levels_mmol": 5.0,
            "bile_production_ml_day": 600.0,
            "bilirubin_mg_dl": 0.8,
            "ammonia_clearance_percent": 100.0,
            "ammonia_blood_level_umol_l": 30.0,
            "alt_u_l": 25.0,
            "ast_u_l": 30.0,
            "alkaline_phosphatase_u_l": 70.0,
            "ggt_u_l": 30.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all hepatic state variables."""
        return [
            "mass_g", "hepatocyte_count_billion", "sinusoidal_perfusion_percent",
            "hepatic_blood_flow_ml_min", "albumin_g_dl", "prothrombin_time_sec", "inr",
            "clotting_factor_production_percent", "glucose_production_mg_dl_hr",
            "glycogen_storage_g", "gluconeogenesis_rate_percent", "lipid_metabolism_percent",
            "cholesterol_synthesis_mg_day", "cytochrome_p450_activity_percent",
            "phase_1_metabolism_percent", "phase_2_conjugation_percent",
            "glutathione_levels_mmol", "bile_production_ml_day", "bilirubin_mg_dl",
            "ammonia_clearance_percent", "ammonia_blood_level_umol_l",
            "alt_u_l", "ast_u_l", "alkaline_phosphatase_u_l", "ggt_u_l"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_cirrhosis(self, child_pugh_score: str = "B"):
        """
        Transition to cirrhosis (end-stage liver disease).
        
        Args:
            child_pugh_score: "A" (mild), "B" (moderate), "C" (severe)
        """
        severity_map = {"A": 0.7, "B": 0.4, "C": 0.2}
        function_factor = severity_map.get(child_pugh_score, 0.4)
        
        # Structural damage
        self.hepatocyte_count_billion = 240.0 * function_factor
        self.sinusoidal_perfusion_percent = 100.0 * function_factor
        
        # Synthetic dysfunction
        self.albumin_g_dl = 2.0 + (function_factor * 2.0)
        self.inr = 1.0 + ((1 - function_factor) * 2.0)
        self.prothrombin_time_sec = 12.0 + ((1 - function_factor) * 15.0)
        self.clotting_factor_production_percent = function_factor * 100.0
        
        # Metabolic dysfunction
        self.glucose_production_mg_dl_hr = 200.0 * function_factor
        self.glycogen_storage_g = 100.0 * function_factor
        self.lipid_metabolism_percent = function_factor * 100.0
        
        # Detoxification impairment
        self.cytochrome_p450_activity_percent = function_factor * 100.0
        self.ammonia_clearance_percent = function_factor * 100.0
        self.ammonia_blood_level_umol_l = 30.0 + ((1 - function_factor) * 120.0)
        self.glutathione_levels_mmol = 5.0 * function_factor
        
        # Excretion dysfunction
        self.bile_production_ml_day = 600.0 * function_factor
        self.bilirubin_mg_dl = 0.8 + ((1 - function_factor) * 10.0)
        
        # Biomarkers
        self.alt_u_l = 25.0 + ((1 - function_factor) * 200.0)
        self.ast_u_l = 30.0 + ((1 - function_factor) * 300.0)
        
        self.state_classification = "CIRRHOTIC"
    
    def transition_to_fatty_liver(self, steatosis_percent: float = 30.0):
        """
        Transition to non-alcoholic fatty liver disease (NAFLD).
        
        Args:
            steatosis_percent: Percentage of hepatocytes with fat (5-100%)
        """
        impairment = steatosis_percent / 100.0
        
        # Metabolic dysfunction
        self.lipid_metabolism_percent = 100.0 * (1 - impairment * 0.5)
        self.glucose_production_mg_dl_hr = 200.0 * (1 + impairment * 0.3)  # Increased
        
        # Mild inflammation
        self.alt_u_l = 25.0 + (impairment * 100.0)
        self.ast_u_l = 30.0 + (impairment * 80.0)
        
        # Oxidative stress
        self.glutathione_levels_mmol = 5.0 * (1 - impairment * 0.4)
        
        if steatosis_percent > 50:
            self.state_classification = "STRESSED"
        else:
            self.state_classification = "COMPENSATED"
    
    def transition_to_acute_hepatitis(self, cause: str = "viral"):
        """
        Transition to acute hepatitis.
        
        Args:
            cause: "viral", "drug_induced", "autoimmune"
        """
        # Severe inflammation
        self.alt_u_l = 500.0
        self.ast_u_l = 400.0
        
        # Synthetic dysfunction
        self.albumin_g_dl = 3.0
        self.inr = 1.5
        self.clotting_factor_production_percent = 70.0
        
        # Detoxification impairment
        self.cytochrome_p450_activity_percent = 60.0
        self.ammonia_clearance_percent = 70.0
        
        # Cholestasis
        self.bilirubin_mg_dl = 5.0
        self.alkaline_phosphatase_u_l = 200.0
        
        self.state_classification = "STRESSED"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Hepatic encephalopathy affects brain
        if self.ammonia_clearance_percent < 50:
            effects["Brain"] = {
                "ammonia_toxicity": (100 - self.ammonia_clearance_percent) / 100.0,
                "consciousness_impairment": (100 - self.ammonia_clearance_percent) / 200.0
            }
        
        # Reduced bile affects gut
        if self.bile_production_ml_day < 300:
            effects["Gut"] = {
                "fat_absorption_percent": -40.0
            }
        
        # Coagulopathy affects blood
        if self.inr > 1.5:
            effects["Blood"] = {
                "bleeding_risk": (self.inr - 1.0) / 2.0
            }
        
        return effects
    
    def receive_from_heart(self, cardiac_output_l_min: float):
        """
        Receive blood flow from heart.
        
        Args:
            cardiac_output_l_min: Cardiac output
        """
        # Liver receives ~25% of cardiac output
        self.hepatic_blood_flow_ml_min = cardiac_output_l_min * 250.0
        
        # Low perfusion impairs function
        if self.hepatic_blood_flow_ml_min < 1000:
            perfusion_factor = self.hepatic_blood_flow_ml_min / 1500.0
            self.sinusoidal_perfusion_percent = perfusion_factor * 100.0
    
    def receive_from_pancreas(self, insulin_level: float, glucagon_level: float):
        """
        Receive hormonal signals from pancreas.
        
        Args:
            insulin_level: Insulin (μU/mL)
            glucagon_level: Glucagon (pg/mL)
        """
        # Insulin promotes glycogen storage
        self.glycogen_storage_g = 100.0 * (insulin_level / 10.0)
        
        # Glucagon promotes glucose production
        self.glucose_production_mg_dl_hr = 200.0 * (glucagon_level / 50.0)
