"""
Kidney (Renal) State Machine.

Models complete renal physiology including:
- Filtration (GFR)
- Tubular function (reabsorption, secretion)
- Endocrine function (renin, EPO, vitamin D)
- Electrolyte balance
- Disease states (CKD stages, AKI)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class KidneyState(OrganState):
    """Renal state machine."""
    
    organ_name: str = "Kidneys"
    
    # Structural parameters
    nephron_count_million: float = 2.0
    glomerular_surface_area_m2: float = 1.5
    
    # Filtration parameters
    gfr_ml_min: float = 100.0  # Glomerular filtration rate
    renal_blood_flow_ml_min: float = 1200.0
    filtration_fraction_percent: float = 20.0
    
    # Tubular function
    sodium_reabsorption_percent: float = 99.5
    water_reabsorption_percent: float = 99.0
    potassium_secretion_meq_day: float = 80.0
    acid_excretion_meq_day: float = 70.0
    
    # Endocrine function
    renin_secretion_ng_ml_hr: float = 1.0
    erythropoietin_mu_ml: float = 10.0
    vitamin_d_activation_percent: float = 100.0
    
    # Excretion
    urine_output_ml_day: float = 1500.0
    creatinine_clearance_ml_min: float = 100.0
    urea_clearance_ml_min: float = 70.0
    
    # Biomarkers
    serum_creatinine_mg_dl: float = 1.0
    bun_mg_dl: float = 15.0  # Blood urea nitrogen
    egfr_ml_min_1_73m2: float = 100.0  # Estimated GFR
    
    # Electrolytes
    serum_sodium_meq_l: float = 140.0
    serum_potassium_meq_l: float = 4.0
    serum_calcium_mg_dl: float = 9.5
    serum_phosphate_mg_dl: float = 3.5
    
    # Volume status
    fluid_retention_ml: float = 0.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal renal state."""
        return {
            "nephron_count_million": 2.0,
            "glomerular_surface_area_m2": 1.5,
            "gfr_ml_min": 100.0,
            "renal_blood_flow_ml_min": 1200.0,
            "filtration_fraction_percent": 20.0,
            "sodium_reabsorption_percent": 99.5,
            "water_reabsorption_percent": 99.0,
            "potassium_secretion_meq_day": 80.0,
            "acid_excretion_meq_day": 70.0,
            "renin_secretion_ng_ml_hr": 1.0,
            "erythropoietin_mu_ml": 10.0,
            "vitamin_d_activation_percent": 100.0,
            "urine_output_ml_day": 1500.0,
            "creatinine_clearance_ml_min": 100.0,
            "urea_clearance_ml_min": 70.0,
            "serum_creatinine_mg_dl": 1.0,
            "bun_mg_dl": 15.0,
            "egfr_ml_min_1_73m2": 100.0,
            "serum_sodium_meq_l": 140.0,
            "serum_potassium_meq_l": 4.0,
            "serum_calcium_mg_dl": 9.5,
            "serum_phosphate_mg_dl": 3.5,
            "fluid_retention_ml": 0.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all renal state variables."""
        return [
            "nephron_count_million", "glomerular_surface_area_m2", "gfr_ml_min",
            "renal_blood_flow_ml_min", "filtration_fraction_percent",
            "sodium_reabsorption_percent", "water_reabsorption_percent",
            "potassium_secretion_meq_day", "acid_excretion_meq_day",
            "renin_secretion_ng_ml_hr", "erythropoietin_mu_ml",
            "vitamin_d_activation_percent", "urine_output_ml_day",
            "creatinine_clearance_ml_min", "urea_clearance_ml_min",
            "serum_creatinine_mg_dl", "bun_mg_dl", "egfr_ml_min_1_73m2",
            "serum_sodium_meq_l", "serum_potassium_meq_l",
            "serum_calcium_mg_dl", "serum_phosphate_mg_dl", "fluid_retention_ml"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_ckd(self, stage: int = 3):
        """
        Transition to chronic kidney disease.
        
        Args:
            stage: CKD stage (1-5)
                Stage 1: GFR ≥90 (kidney damage with normal GFR)
                Stage 2: GFR 60-89 (mild reduction)
                Stage 3: GFR 30-59 (moderate reduction)
                Stage 4: GFR 15-29 (severe reduction)
                Stage 5: GFR <15 (kidney failure)
        """
        stage_map = {
            1: 95.0,
            2: 75.0,
            3: 45.0,
            4: 22.0,
            5: 10.0
        }
        
        target_gfr = stage_map.get(stage, 45.0)
        reduction_factor = target_gfr / 100.0
        
        # Filtration impairment
        self.gfr_ml_min = target_gfr
        self.egfr_ml_min_1_73m2 = target_gfr
        self.creatinine_clearance_ml_min = target_gfr
        self.nephron_count_million = 2.0 * reduction_factor
        
        # Biomarkers
        self.serum_creatinine_mg_dl = 1.0 / reduction_factor
        self.bun_mg_dl = 15.0 + ((1 - reduction_factor) * 85.0)
        
        # Endocrine dysfunction
        self.erythropoietin_mu_ml = 10.0 * reduction_factor
        self.vitamin_d_activation_percent = 100.0 * reduction_factor
        
        # Electrolyte imbalance
        if stage >= 4:
            self.serum_potassium_meq_l = 4.0 + ((1 - reduction_factor) * 2.0)
            self.serum_phosphate_mg_dl = 3.5 + ((1 - reduction_factor) * 3.5)
            self.serum_calcium_mg_dl = 9.5 - ((1 - reduction_factor) * 1.5)
        
        # Fluid retention
        self.fluid_retention_ml = (1 - reduction_factor) * 2000.0
        self.urine_output_ml_day = 1500.0 * reduction_factor
        
        if stage >= 4:
            self.state_classification = "FAILING"
        elif stage >= 3:
            self.state_classification = "COMPENSATED"
        else:
            self.state_classification = "STRESSED"
    
    def transition_to_aki(self, severity: str = "moderate"):
        """
        Transition to acute kidney injury.
        
        Args:
            severity: "mild", "moderate", "severe"
        """
        severity_map = {
            "mild": 0.7,
            "moderate": 0.5,
            "severe": 0.2
        }
        
        function_factor = severity_map.get(severity, 0.5)
        
        # Acute filtration failure
        self.gfr_ml_min = 100.0 * function_factor
        self.urine_output_ml_day = 1500.0 * function_factor
        
        # Rapid biomarker rise
        self.serum_creatinine_mg_dl = 1.0 + ((1 - function_factor) * 5.0)
        self.bun_mg_dl = 15.0 + ((1 - function_factor) * 85.0)
        
        # Electrolyte chaos
        self.serum_potassium_meq_l = 4.0 + ((1 - function_factor) * 3.0)
        
        # Fluid overload
        self.fluid_retention_ml = (1 - function_factor) * 3000.0
        
        self.state_classification = "STRESSED"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Fluid retention affects heart
        if self.fluid_retention_ml > 1000:
            effects["Heart"] = {
                "preload_mmhg": self.fluid_retention_ml / 100.0,
                "bnp_pg_ml": self.fluid_retention_ml / 20.0
            }
        
        # Reduced EPO affects blood
        if self.erythropoietin_mu_ml < 5:
            effects["Blood"] = {
                "hemoglobin_g_dl": -(10 - self.erythropoietin_mu_ml) / 2.0
            }
        
        # Reduced vitamin D affects bones
        if self.vitamin_d_activation_percent < 60:
            effects["Bones"] = {
                "bone_mineral_density_percent": -(100 - self.vitamin_d_activation_percent) / 5.0
            }
        
        return effects
    
    def receive_from_heart(self, cardiac_output_l_min: float, blood_pressure_systolic: float):
        """
        Receive blood flow and pressure from heart.
        
        Args:
            cardiac_output_l_min: Cardiac output
            blood_pressure_systolic: Systolic BP (mmHg)
        """
        # Kidneys receive ~20% of cardiac output
        self.renal_blood_flow_ml_min = cardiac_output_l_min * 240.0
        
        # GFR depends on renal perfusion pressure
        pressure_factor = blood_pressure_systolic / 120.0
        flow_factor = self.renal_blood_flow_ml_min / 1200.0
        
        self.gfr_ml_min = 100.0 * pressure_factor * flow_factor
        self.egfr_ml_min_1_73m2 = self.gfr_ml_min
