"""
Immune System State Machine.

Models complete immune physiology including:
- Innate immunity (neutrophils, macrophages, NK cells)
- Adaptive immunity (T cells, B cells, antibodies)
- Inflammation and cytokines
- Disease states (autoimmune, immunodeficiency, chronic inflammation)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class ImmuneState(OrganState):
    """Immune system state machine."""
    
    organ_name: str = "Immune System"
    
    # Innate immunity
    neutrophil_count_per_ul: float = 4000.0
    macrophage_activity_percent: float = 100.0
    nk_cell_count_per_ul: float = 200.0
    complement_activity_percent: float = 100.0
    
    # Adaptive immunity
    t_cell_count_per_ul: float = 1000.0
    b_cell_count_per_ul: float = 200.0
    cd4_count_per_ul: float = 700.0
    cd8_count_per_ul: float = 300.0
    cd4_cd8_ratio: float = 2.3
    
    # Antibodies
    igg_mg_dl: float = 1000.0
    iga_mg_dl: float = 200.0
    igm_mg_dl: float = 100.0
    ige_iu_ml: float = 50.0
    
    # Cytokines and inflammation
    il6_pg_ml: float = 5.0  # Interleukin-6
    tnf_alpha_pg_ml: float = 10.0  # Tumor necrosis factor
    il10_pg_ml: float = 5.0  # Anti-inflammatory
    crp_mg_l: float = 1.0  # C-reactive protein
    esr_mm_hr: float = 10.0  # Erythrocyte sedimentation rate
    
    # Immune function
    phagocytic_capacity_percent: float = 100.0
    antibody_production_percent: float = 100.0
    cytotoxic_activity_percent: float = 100.0
    
    # Systemic inflammation
    systemic_inflammation: float = 0.0  # 0-1 scale
    inflammatory_state: str = "NORMAL"  # NORMAL, ACUTE, CHRONIC
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal immune state."""
        return {
            "neutrophil_count_per_ul": 4000.0,
            "macrophage_activity_percent": 100.0,
            "nk_cell_count_per_ul": 200.0,
            "complement_activity_percent": 100.0,
            "t_cell_count_per_ul": 1000.0,
            "b_cell_count_per_ul": 200.0,
            "cd4_count_per_ul": 700.0,
            "cd8_count_per_ul": 300.0,
            "cd4_cd8_ratio": 2.3,
            "igg_mg_dl": 1000.0,
            "iga_mg_dl": 200.0,
            "igm_mg_dl": 100.0,
            "ige_iu_ml": 50.0,
            "il6_pg_ml": 5.0,
            "tnf_alpha_pg_ml": 10.0,
            "il10_pg_ml": 5.0,
            "crp_mg_l": 1.0,
            "esr_mm_hr": 10.0,
            "phagocytic_capacity_percent": 100.0,
            "antibody_production_percent": 100.0,
            "cytotoxic_activity_percent": 100.0,
            "systemic_inflammation": 0.0,
            "inflammatory_state": "NORMAL",
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all immune state variables."""
        return [
            "neutrophil_count_per_ul", "macrophage_activity_percent",
            "nk_cell_count_per_ul", "complement_activity_percent",
            "t_cell_count_per_ul", "b_cell_count_per_ul",
            "cd4_count_per_ul", "cd8_count_per_ul", "cd4_cd8_ratio",
            "igg_mg_dl", "iga_mg_dl", "igm_mg_dl", "ige_iu_ml",
            "il6_pg_ml", "tnf_alpha_pg_ml", "il10_pg_ml",
            "crp_mg_l", "esr_mm_hr",
            "phagocytic_capacity_percent", "antibody_production_percent",
            "cytotoxic_activity_percent", "systemic_inflammation"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_autoimmune(self, autoimmune_type: str = "rheumatoid_arthritis"):
        """
        Transition to autoimmune disease.
        
        Args:
            autoimmune_type: "rheumatoid_arthritis", "lupus", "multiple_sclerosis"
        """
        # Hyperactive immune response
        self.t_cell_count_per_ul = 1500.0  # ↑50%
        self.b_cell_count_per_ul = 350.0  # ↑75%
        
        # Autoantibody production
        self.igg_mg_dl = 1800.0  # ↑80%
        self.antibody_production_percent = 180.0
        
        # Chronic inflammation
        self.il6_pg_ml = 50.0  # ↑900%
        self.tnf_alpha_pg_ml = 80.0  # ↑700%
        self.crp_mg_l = 25.0  # ↑2400%
        self.esr_mm_hr = 50.0  # ↑400%
        
        # Systemic inflammation
        self.systemic_inflammation = 0.7
        self.inflammatory_state = "CHRONIC"
        
        self.state_classification = "AUTOIMMUNE"
    
    def transition_to_immunodeficiency(self, severity: str = "moderate"):
        """
        Transition to immunodeficiency.
        
        Args:
            severity: "mild", "moderate", "severe"
        """
        severity_map = {
            "mild": 0.7,
            "moderate": 0.5,
            "severe": 0.2
        }
        
        function_factor = severity_map.get(severity, 0.5)
        
        # Reduced cell counts
        self.neutrophil_count_per_ul = 4000.0 * function_factor
        self.t_cell_count_per_ul = 1000.0 * function_factor
        self.cd4_count_per_ul = 700.0 * function_factor
        self.b_cell_count_per_ul = 200.0 * function_factor
        
        # Reduced antibodies
        self.igg_mg_dl = 1000.0 * function_factor
        self.iga_mg_dl = 200.0 * function_factor
        
        # Impaired function
        self.phagocytic_capacity_percent = 100.0 * function_factor
        self.antibody_production_percent = 100.0 * function_factor
        self.cytotoxic_activity_percent = 100.0 * function_factor
        
        self.state_classification = "IMMUNODEFICIENT"
    
    def transition_to_chronic_inflammation(self, inflammation_level: float = 0.6):
        """
        Transition to chronic low-grade inflammation.
        
        Args:
            inflammation_level: Inflammation intensity (0-1 scale)
        """
        # Elevated inflammatory markers
        self.il6_pg_ml = 5.0 + (inflammation_level * 45.0)
        self.tnf_alpha_pg_ml = 10.0 + (inflammation_level * 70.0)
        self.crp_mg_l = 1.0 + (inflammation_level * 15.0)
        self.esr_mm_hr = 10.0 + (inflammation_level * 40.0)
        
        # Reduced anti-inflammatory
        self.il10_pg_ml = 5.0 * (1 - inflammation_level * 0.5)
        
        # Systemic effects
        self.systemic_inflammation = inflammation_level
        self.inflammatory_state = "CHRONIC"
        
        self.state_classification = "INFLAMED"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Chronic inflammation affects all organs
        if self.systemic_inflammation > 0.5:
            effects["Heart"] = {
                "inflammatory_damage": self.systemic_inflammation * 20.0
            }
            effects["Brain"] = {
                "neuroinflammation": self.systemic_inflammation
            }
            effects["Liver"] = {
                "inflammatory_stress": self.systemic_inflammation * 30.0
            }
        
        # Immunodeficiency affects infection resistance
        if self.phagocytic_capacity_percent < 50:
            effects["Lungs"] = {
                "infection_risk": (100 - self.phagocytic_capacity_percent) / 100.0
            }
        
        return effects
    
    def receive_from_gut(self, intestinal_permeability: float):
        """
        Receive gut barrier status.
        
        Args:
            intestinal_permeability: Gut permeability index (1.0 = normal)
        """
        # Leaky gut triggers immune activation
        if intestinal_permeability > 2.0:
            activation = (intestinal_permeability - 1.0) / 4.0
            self.systemic_inflammation = activation
            self.il6_pg_ml = 5.0 + (activation * 45.0)
            self.crp_mg_l = 1.0 + (activation * 15.0)
