"""
Heart (Cardiac) State Machine.

Models complete cardiac physiology including:
- Hemodynamics (HR, SV, CO, BP)
- Electrical conduction
- Metabolic function
- Perfusion
- Disease states (heart failure, hypertrophy, ischemia)
"""

from dataclasses import dataclass, field
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class HeartState(OrganState):
    """Cardiac state machine."""
    
    organ_name: str = "Heart"
    
    # Structural parameters
    mass_g: float = 300.0
    wall_thickness_mm: float = 11.0
    chamber_volume_ml: float = 120.0
    ejection_fraction_percent: float = 60.0
    
    # Hemodynamic parameters
    heart_rate_bpm: float = 70.0
    stroke_volume_ml: float = 70.0
    cardiac_output_l_min: float = 5.0
    contractility_percent: float = 100.0
    preload_mmhg: float = 8.0
    afterload_mmhg: float = 100.0
    
    # Electrical parameters
    pr_interval_ms: float = 160.0
    qrs_duration_ms: float = 100.0
    qt_interval_ms: float = 400.0
    conduction_velocity_m_s: float = 1.0
    
    # Metabolic parameters
    oxygen_consumption_ml_min: float = 30.0
    atp_production_percent: float = 100.0
    lactate_production_mmol_l: float = 1.0
    mitochondrial_density_percent: float = 100.0
    
    # Perfusion parameters
    coronary_blood_flow_ml_min: float = 250.0
    myocardial_oxygen_delivery_percent: float = 100.0
    capillary_density_per_mm2: float = 3000.0
    
    # Biomarkers
    bnp_pg_ml: float = 50.0  # Brain natriuretic peptide
    troponin_ng_ml: float = 0.01  # Cardiac troponin
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal cardiac state."""
        return {
            # Structural
            "mass_g": 300.0,
            "wall_thickness_mm": 11.0,
            "chamber_volume_ml": 120.0,
            "ejection_fraction_percent": 60.0,
            
            # Hemodynamic
            "heart_rate_bpm": 70.0,
            "stroke_volume_ml": 70.0,
            "cardiac_output_l_min": 5.0,
            "contractility_percent": 100.0,
            "preload_mmhg": 8.0,
            "afterload_mmhg": 100.0,
            
            # Electrical
            "pr_interval_ms": 160.0,
            "qrs_duration_ms": 100.0,
            "qt_interval_ms": 400.0,
            "conduction_velocity_m_s": 1.0,
            
            # Metabolic
            "oxygen_consumption_ml_min": 30.0,
            "atp_production_percent": 100.0,
            "lactate_production_mmol_l": 1.0,
            "mitochondrial_density_percent": 100.0,
            
            # Perfusion
            "coronary_blood_flow_ml_min": 250.0,
            "myocardial_oxygen_delivery_percent": 100.0,
            "capillary_density_per_mm2": 3000.0,
            
            # Biomarkers
            "bnp_pg_ml": 50.0,
            "troponin_ng_ml": 0.01,
            
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all cardiac state variables."""
        return [
            "mass_g", "wall_thickness_mm", "chamber_volume_ml", "ejection_fraction_percent",
            "heart_rate_bpm", "stroke_volume_ml", "cardiac_output_l_min", "contractility_percent",
            "preload_mmhg", "afterload_mmhg",
            "pr_interval_ms", "qrs_duration_ms", "qt_interval_ms", "conduction_velocity_m_s",
            "oxygen_consumption_ml_min", "atp_production_percent", "lactate_production_mmol_l",
            "mitochondrial_density_percent",
            "coronary_blood_flow_ml_min", "myocardial_oxygen_delivery_percent",
            "capillary_density_per_mm2",
            "bnp_pg_ml", "troponin_ng_ml"
        ]
    
    def update_cardiac_output(self):
        """Update cardiac output based on HR and SV."""
        self.cardiac_output_l_min = (self.heart_rate_bpm * self.stroke_volume_ml) / 1000.0
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_heart_failure(self, cause: str = "ischemic"):
        """
        Transition to heart failure state.
        
        Args:
            cause: "ischemic", "hypertensive", "dilated"
        """
        self.ejection_fraction_percent = 35.0  # ↓42%
        self.cardiac_output_l_min = 3.5  # ↓30%
        self.stroke_volume_ml = 50.0  # ↓29%
        self.contractility_percent = 50.0  # ↓50%
        
        # Metabolic dysfunction
        self.atp_production_percent = 60.0  # ↓40%
        self.mitochondrial_density_percent = 70.0  # ↓30%
        self.oxygen_consumption_ml_min = 45.0  # ↑50% (inefficient)
        
        # Biomarkers
        self.bnp_pg_ml = 500.0  # ↑900%
        self.troponin_ng_ml = 0.05  # ↑400%
        
        # Structural remodeling
        if cause == "dilated":
            self.chamber_volume_ml = 200.0  # ↑67%
            self.wall_thickness_mm = 9.0  # ↓18%
        elif cause == "hypertensive":
            self.wall_thickness_mm = 15.0  # ↑36%
            self.chamber_volume_ml = 100.0  # ↓17%
        
        self.state_classification = "FAILING"
        self.update_cardiac_output()
    
    def transition_to_hypertrophy(self, hypertension_duration_years: float = 5.0):
        """
        Transition to cardiac hypertrophy (compensated).
        
        Args:
            hypertension_duration_years: Duration of hypertension
        """
        # Compensatory hypertrophy
        self.mass_g = 400.0  # ↑33%
        self.wall_thickness_mm = 15.0  # ↑36%
        self.contractility_percent = 120.0  # ↑20% (compensation)
        
        # Increased workload
        self.afterload_mmhg = 140.0  # ↑40%
        self.oxygen_consumption_ml_min = 50.0  # ↑67%
        
        # Metabolic strain
        self.atp_production_percent = 80.0  # ↓20%
        self.mitochondrial_density_percent = 90.0  # ↓10%
        
        # Biomarkers
        self.bnp_pg_ml = 150.0  # ↑200%
        
        self.state_classification = "COMPENSATED"
        self.update_cardiac_output()
    
    def transition_to_ischemia(self, stenosis_percent: float = 70.0):
        """
        Transition to myocardial ischemia.
        
        Args:
            stenosis_percent: Coronary artery stenosis (0-100%)
        """
        # Reduced perfusion
        reduction_factor = stenosis_percent / 100.0
        self.coronary_blood_flow_ml_min = 250.0 * (1 - reduction_factor)
        self.myocardial_oxygen_delivery_percent = 100.0 * (1 - reduction_factor)
        
        # Metabolic dysfunction
        self.atp_production_percent = 100.0 * (1 - reduction_factor * 0.8)
        self.lactate_production_mmol_l = 1.0 + (stenosis_percent / 10.0)
        
        # Contractility impairment
        self.contractility_percent = 100.0 * (1 - reduction_factor * 0.5)
        self.ejection_fraction_percent = 60.0 * (1 - reduction_factor * 0.4)
        
        # Biomarkers
        if stenosis_percent > 90:  # Acute MI
            self.troponin_ng_ml = 10.0  # ↑100,000%
            self.bnp_pg_ml = 300.0
        else:
            self.troponin_ng_ml = 0.1  # ↑900%
        
        self.state_classification = "STRESSED"
        self.update_cardiac_output()
    
    def transition_to_arrhythmia(self, arrhythmia_type: str = "atrial_fibrillation"):
        """
        Transition to arrhythmic state.
        
        Args:
            arrhythmia_type: "atrial_fibrillation", "ventricular_tachycardia", "heart_block"
        """
        if arrhythmia_type == "atrial_fibrillation":
            self.heart_rate_bpm = 140.0  # ↑100%
            self.stroke_volume_ml = 50.0  # ↓29% (loss of atrial kick)
            self.pr_interval_ms = 0.0  # Absent P waves
            
        elif arrhythmia_type == "ventricular_tachycardia":
            self.heart_rate_bpm = 180.0  # ↑157%
            self.stroke_volume_ml = 40.0  # ↓43%
            self.qrs_duration_ms = 200.0  # ↑100%
            
        elif arrhythmia_type == "heart_block":
            self.heart_rate_bpm = 40.0  # ↓43%
            self.pr_interval_ms = 300.0  # ↑88%
            self.conduction_velocity_m_s = 0.5  # ↓50%
        
        self.state_classification = "STRESSED"
        self.update_cardiac_output()
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Low cardiac output affects kidneys
        if self.cardiac_output_l_min < 4.0:
            effects["Kidneys"] = {
                "renal_blood_flow_ml_min": -300.0,  # ↓25%
                "gfr_ml_min": -20.0  # ↓20%
            }
        
        # Low cardiac output affects brain
        if self.cardiac_output_l_min < 3.5:
            effects["Brain"] = {
                "cerebral_blood_flow_ml_min": -150.0,  # ↓20%
            }
        
        # Low cardiac output affects liver
        if self.cardiac_output_l_min < 4.0:
            effects["Liver"] = {
                "hepatic_blood_flow_ml_min": -300.0,  # ↓20%
            }
        
        return effects
    
    def receive_from_lungs(self, lung_oxygen_saturation: float):
        """
        Receive oxygen delivery from lungs.
        
        Args:
            lung_oxygen_saturation: Arterial O2 saturation (%)
        """
        self.myocardial_oxygen_delivery_percent = lung_oxygen_saturation
        
        # Hypoxia affects ATP production
        if lung_oxygen_saturation < 90:
            hypoxia_factor = lung_oxygen_saturation / 100.0
            self.atp_production_percent = 100.0 * hypoxia_factor
    
    def receive_from_kidneys(self, fluid_retention_ml: float):
        """
        Receive volume status from kidneys.
        
        Args:
            fluid_retention_ml: Excess fluid retained (ml)
        """
        # Fluid retention increases preload
        self.preload_mmhg = 8.0 + (fluid_retention_ml / 100.0)
        
        # Excessive preload can worsen heart failure
        if self.preload_mmhg > 20.0:
            self.bnp_pg_ml += 50.0
    
    def receive_from_nervous_system(self, sympathetic_activity: float):
        """
        Receive autonomic input from nervous system.
        
        Args:
            sympathetic_activity: Sympathetic tone (0-1 scale)
        """
        # Sympathetic activation increases HR and contractility
        self.heart_rate_bpm = 70.0 + (sympathetic_activity * 60.0)
        self.contractility_percent = 100.0 + (sympathetic_activity * 40.0)
        self.update_cardiac_output()
