"""
Gut (Gastrointestinal) State Machine.

Models complete GI physiology including:
- Digestion and absorption
- Microbiome composition
- Barrier function
- Gut-brain axis
- Disease states (IBD, IBS, dysbiosis)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class GutState(OrganState):
    """Gastrointestinal state machine."""
    
    organ_name: str = "Gut"
    
    # Structural parameters
    intestinal_length_m: float = 7.0
    surface_area_m2: float = 250.0  # With villi
    mucosal_thickness_mm: float = 0.5
    
    # Digestion parameters
    gastric_acid_ph: float = 2.0
    bile_salt_concentration_mmol_l: float = 10.0
    pancreatic_enzyme_activity_percent: float = 100.0
    digestive_capacity_percent: float = 100.0
    
    # Absorption parameters
    nutrient_absorption_percent: float = 95.0
    fat_absorption_percent: float = 95.0
    protein_absorption_percent: float = 95.0
    carbohydrate_absorption_percent: float = 95.0
    water_absorption_l_day: float = 8.0
    
    # Microbiome parameters
    bacterial_count_trillion: float = 100.0
    microbiome_diversity_index: float = 100.0
    beneficial_bacteria_percent: float = 85.0
    pathogenic_bacteria_percent: float = 5.0
    
    # Barrier function
    intestinal_permeability_index: float = 1.0  # 1.0 = normal
    tight_junction_integrity_percent: float = 100.0
    mucus_layer_thickness_um: float = 100.0
    
    # Immune function
    gut_associated_lymphoid_tissue_percent: float = 100.0
    secretory_iga_mg_dl: float = 200.0
    inflammatory_cytokines_pg_ml: float = 10.0
    
    # Motility
    transit_time_hours: float = 24.0
    peristalsis_frequency_per_min: float = 3.0
    
    # Metabolic/endocrine
    serotonin_production_percent: float = 100.0  # 90% of body's serotonin
    short_chain_fatty_acids_mmol_l: float = 100.0
    
    # Perfusion
    splanchnic_blood_flow_ml_min: float = 1400.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal GI state."""
        return {
            "intestinal_length_m": 7.0,
            "surface_area_m2": 250.0,
            "mucosal_thickness_mm": 0.5,
            "gastric_acid_ph": 2.0,
            "bile_salt_concentration_mmol_l": 10.0,
            "pancreatic_enzyme_activity_percent": 100.0,
            "digestive_capacity_percent": 100.0,
            "nutrient_absorption_percent": 95.0,
            "fat_absorption_percent": 95.0,
            "protein_absorption_percent": 95.0,
            "carbohydrate_absorption_percent": 95.0,
            "water_absorption_l_day": 8.0,
            "bacterial_count_trillion": 100.0,
            "microbiome_diversity_index": 100.0,
            "beneficial_bacteria_percent": 85.0,
            "pathogenic_bacteria_percent": 5.0,
            "intestinal_permeability_index": 1.0,
            "tight_junction_integrity_percent": 100.0,
            "mucus_layer_thickness_um": 100.0,
            "gut_associated_lymphoid_tissue_percent": 100.0,
            "secretory_iga_mg_dl": 200.0,
            "inflammatory_cytokines_pg_ml": 10.0,
            "transit_time_hours": 24.0,
            "peristalsis_frequency_per_min": 3.0,
            "serotonin_production_percent": 100.0,
            "short_chain_fatty_acids_mmol_l": 100.0,
            "splanchnic_blood_flow_ml_min": 1400.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all GI state variables."""
        return [
            "intestinal_length_m", "surface_area_m2", "mucosal_thickness_mm",
            "gastric_acid_ph", "bile_salt_concentration_mmol_l",
            "pancreatic_enzyme_activity_percent", "digestive_capacity_percent",
            "nutrient_absorption_percent", "fat_absorption_percent",
            "protein_absorption_percent", "carbohydrate_absorption_percent",
            "water_absorption_l_day", "bacterial_count_trillion",
            "microbiome_diversity_index", "beneficial_bacteria_percent",
            "pathogenic_bacteria_percent", "intestinal_permeability_index",
            "tight_junction_integrity_percent", "mucus_layer_thickness_um",
            "gut_associated_lymphoid_tissue_percent", "secretory_iga_mg_dl",
            "inflammatory_cytokines_pg_ml", "transit_time_hours",
            "peristalsis_frequency_per_min", "serotonin_production_percent",
            "short_chain_fatty_acids_mmol_l", "splanchnic_blood_flow_ml_min"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_ibd(self, severity: str = "moderate"):
        """
        Transition to inflammatory bowel disease (Crohn's or UC).
        
        Args:
            severity: "mild", "moderate", "severe"
        """
        severity_map = {
            "mild": 0.7,
            "moderate": 0.5,
            "severe": 0.3
        }
        
        function_factor = severity_map.get(severity, 0.5)
        
        # Inflammation
        self.inflammatory_cytokines_pg_ml = 10.0 + ((1 - function_factor) * 190.0)
        
        # Barrier dysfunction
        self.intestinal_permeability_index = 1.0 + ((1 - function_factor) * 4.0)
        self.tight_junction_integrity_percent = 100.0 * function_factor
        self.mucus_layer_thickness_um = 100.0 * function_factor
        
        # Absorption impairment
        self.nutrient_absorption_percent = 95.0 * function_factor
        self.water_absorption_l_day = 8.0 * function_factor
        
        # Immune dysfunction
        self.gut_associated_lymphoid_tissue_percent = 100.0 * (1 + (1 - function_factor) * 0.5)
        
        # Microbiome disruption
        self.microbiome_diversity_index = 100.0 * function_factor
        self.beneficial_bacteria_percent = 85.0 * function_factor
        
        self.state_classification = "INFLAMED"
    
    def transition_to_dysbiosis(self, dysbiosis_severity: float = 50.0):
        """
        Transition to microbiome dysbiosis.
        
        Args:
            dysbiosis_severity: Percentage of microbiome imbalance (0-100%)
        """
        imbalance = dysbiosis_severity / 100.0
        
        # Microbiome changes
        self.microbiome_diversity_index = 100.0 * (1 - imbalance * 0.6)
        self.beneficial_bacteria_percent = 85.0 * (1 - imbalance * 0.5)
        self.pathogenic_bacteria_percent = 5.0 + (imbalance * 30.0)
        
        # Metabolic changes
        self.short_chain_fatty_acids_mmol_l = 100.0 * (1 - imbalance * 0.4)
        self.serotonin_production_percent = 100.0 * (1 - imbalance * 0.3)
        
        # Barrier effects
        self.intestinal_permeability_index = 1.0 + (imbalance * 2.0)
        
        # Mild inflammation
        self.inflammatory_cytokines_pg_ml = 10.0 + (imbalance * 40.0)
        
        self.state_classification = "DYSBIOTIC"
    
    def transition_to_leaky_gut(self):
        """Transition to increased intestinal permeability (leaky gut)."""
        # Barrier breakdown
        self.intestinal_permeability_index = 3.0
        self.tight_junction_integrity_percent = 40.0
        self.mucus_layer_thickness_um = 50.0
        
        # Inflammation
        self.inflammatory_cytokines_pg_ml = 80.0
        
        # Immune activation
        self.gut_associated_lymphoid_tissue_percent = 150.0
        
        self.state_classification = "PERMEABLE"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Gut-brain axis (serotonin)
        if self.serotonin_production_percent < 70:
            effects["Brain"] = {
                "serotonin_nmol_g": -(100 - self.serotonin_production_percent) / 200.0
            }
        
        # Leaky gut affects immune system
        if self.intestinal_permeability_index > 2.0:
            effects["Immune"] = {
                "systemic_inflammation": (self.intestinal_permeability_index - 1.0) / 4.0
            }
        
        # Malabsorption affects nutrition
        if self.nutrient_absorption_percent < 70:
            effects["Muscles"] = {
                "protein_synthesis_percent": -(95 - self.nutrient_absorption_percent)
            }
        
        return effects
    
    def receive_from_liver(self, bile_production_ml_day: float):
        """
        Receive bile from liver.
        
        Args:
            bile_production_ml_day: Bile production rate
        """
        # Bile affects fat digestion
        self.bile_salt_concentration_mmol_l = bile_production_ml_day / 60.0
        
        if bile_production_ml_day < 300:
            bile_factor = bile_production_ml_day / 600.0
            self.fat_absorption_percent = 95.0 * bile_factor
    
    def receive_from_pancreas(self, enzyme_secretion_ml_day: float):
        """
        Receive pancreatic enzymes.
        
        Args:
            enzyme_secretion_ml_day: Enzyme secretion rate
        """
        self.pancreatic_enzyme_activity_percent = (enzyme_secretion_ml_day / 1500.0) * 100.0
        
        # Affects overall digestion
        self.digestive_capacity_percent = self.pancreatic_enzyme_activity_percent
