"""
Geriatric Medicine Module.

Implements specialized geriatric physiology:
1. Frailty Index calculation.
2. Age-related physiological decline (Senescence).
3. Beers Criteria and polypharmacy risk assessment.
"""

from typing import Dict, List, Any
from body_state_model import HumanBodyState

class GeriatricSpecialist:
    """Specialist for geriatric physiological assessment."""
    
    @staticmethod
    def calculate_frailty_index(body: HumanBodyState) -> float:
        """
        Calculate Fried's Frailty Phenotype / Rockwood Index.
        Based on physiological deficits across the 12-organ model.
        
        Returns score from 0.0 (Robust) to 1.0 (Severely Frail).
        """
        deficits = 0
        total_parameters = 0
        
        # Check specific geriatric biomarkers
        # 1. Muscle Mass (Sarcopenia)
        if body.muscles.muscle_mass_kg < 20.0:
            deficits += 1
        total_parameters += 1
        
        # 2. Bone Density (Osteoporosis)
        if body.bones.bone_mineral_density_percent < 70.0:
            deficits += 1
        total_parameters += 1
        
        # 3. Cognitive Function (Dementia risk)
        if body.brain.memory_score < 80.0:
            deficits += 1
        total_parameters += 1
        
        # 4. Kidney Function (eGFR < 60)
        if body.kidneys.gfr_ml_min < 60.0:
            deficits += 1
        total_parameters += 1
        
        # 5. Immune Reserve (Low baseline cytokines/response)
        if body.immune_system.phagocytic_capacity_percent < 60.0:
            deficits += 1
        total_parameters += 1
        
        # 6. Cardiac Reserve (Ejection Fraction)
        if body.heart.ejection_fraction_percent < 50.0:
            deficits += 1
        total_parameters += 1
        
        # 7. Lung Function (FEV1)
        if body.lungs.fev1_percent_predicted < 70.0:
            deficits += 1
        total_parameters += 1

        # 8. Nutritonal / Gut (Low absorption)
        if body.gut.nutrient_absorption_percent < 75.0:
            deficits += 1
        total_parameters += 1
        
        return deficits / total_parameters

    @staticmethod
    def apply_senescence(body: HumanBodyState, age_years: float, senescence_rate_multiplier: float = 1.0):
        """
        Apply age-related physiological degradation.
        
        Args:
            body: HumanBodyState instance
            age_years: Chronological age
            senescence_rate_multiplier: Rate of aging (1.0 = normal, < 1.0 = geroprotection)
        """
        aging_factor = max(0, (age_years - 25) / 75.0) * senescence_rate_multiplier # Baseline is age 25
        
        # 1. Kidneys: GFR falls ~0.75-1.0 mL/min per year after age 40
        if age_years > 40:
            reduction = (age_years - 40) * 0.85 * senescence_rate_multiplier
            body.kidneys.gfr_ml_min = max(30, 100 - reduction)
            
        # 2. Muscles: Sarcopenia (1-2% loss of mass per year after age 50)
        basal_muscles = body.muscles.get_basal_state()
        if age_years > 50:
            muscle_reduction = (age_years - 50) * 0.015 * senescence_rate_multiplier
            body.muscles.muscle_mass_kg = basal_muscles["muscle_mass_kg"] * (1 - muscle_reduction)
            body.muscles.max_strength_percent = basal_muscles["max_strength_percent"] * (1 - muscle_reduction)
            
        # 3. Brain: Volume loss and slowing
        basal_brain = body.brain.get_basal_state()
        brain_reduction = aging_factor * 0.2
        body.brain.neuron_count_billion = basal_brain["neuron_count_billion"] * (1 - brain_reduction)
        body.brain.processing_speed_ms = basal_brain["processing_speed_ms"] + (aging_factor * 100.0)
        
        # 4. Heart: Reduced reserve and increased afterload
        basal_heart = body.heart.get_basal_state()
        body.heart.afterload_mmhg = basal_heart["afterload_mmhg"] * (1 + aging_factor * 0.3)
        
        # 5. Skin: Thinning and reduced elasticity
        basal_skin = body.skin.get_basal_state()
        body.skin.thickness_mm = basal_skin["thickness_mm"] * (1 - aging_factor * 0.4)
        body.skin.collagen_content_percent = basal_skin["collagen_content_percent"] * (1 - aging_factor * 0.6)
        
        # 6. Bones: Bone Mineral Density loss
        if age_years > 50:
            basal_bones = body.bones.get_basal_state()
            bone_loss = (age_years - 50) * 0.005 * senescence_rate_multiplier
            body.bones.bone_mineral_density_percent = basal_bones["bone_mineral_density_percent"] * (1 - bone_loss)

    @staticmethod
    def check_beers_criteria(drug_name: str, body: HumanBodyState) -> List[Dict]:
        """
        Check if drug is potentially inappropriate for elderly (PIM).
        Based on AGS Beers Criteria 2023.
        """
        # PIM list (Potentially Inappropriate Medications)
        pims = {
            "Alprazolam": "Benzodiazepines: High risk of falls, hip fractures, and delirium in elderly.",
            "Lorazepam": "Benzodiazepines: High risk of falls, delirium, and cognitive impairment.",
            "Zolpidem": "Z-drugs: Increased risk of falls and fractures; minimal improvement in sleep latency.",
            "Ibuprofen": "NSAIDs: Increased risk of GI bleeding and AKI in elderly with reduced GFR.",
            "Naproxen": "NSAIDs: Higher risk of GI bleeding and renal impairment.",
            "Digoxin": "Cardiac Glycosides: Higher toxicity risk; avoid as first-line for Afib/HF.",
            "Amitriptyline": "TCAs: Highly anticholinergic; risks of orthostatic hypotension and sedation.",
            "Olanzapine": "Antipsychotics: Increased risk of stroke and mortality in patients with dementia."
        }
        
        warnings = []
        for pim_name, reason in pims.items():
            if pim_name.lower() in drug_name.lower():
                warnings.append({
                    "type": "BEERS_CRITERIA_PIM",
                    "drug": pim_name,
                    "reason": reason,
                    "severity": "high"
                })
        
        # 2. Condition-specific warnings
        # NSAIDs in CKD
        if "Ibuprofen" in drug_name or "Naproxen" in drug_name:
            if body.kidneys.gfr_ml_min < 60.0:
                warnings.append({
                    "type": "CONDITION_SPECIFIC_CONTRAINDICATION",
                    "reason": "NSAIDs contraindicated/cautioned in elderly with GFR < 60 due to renal failure risk.",
                    "severity": "critical"
                })
                
        return warnings
