"""
Endocrine System State Machine.

Models hormonal regulation including:
- Thyroid axis (TSH, T3, T4)
- Adrenal axis (cortisol, ACTH)
- Reproductive hormones
- Growth hormone
- Disease states (hypothyroidism, Cushing's, PCOS)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class EndocrineState(OrganState):
    """Endocrine system state machine."""
    
    organ_name: str = "Endocrine System"
    
    # Thyroid axis
    tsh_mu_ml: float = 2.0  # Thyroid stimulating hormone
    t4_ug_dl: float = 8.0  # Thyroxine
    t3_ng_dl: float = 120.0  # Triiodothyronine
    thyroid_function_percent: float = 100.0
    
    # Adrenal axis
    cortisol_ug_dl: float = 12.0
    acth_pg_ml: float = 25.0  # Adrenocorticotropic hormone
    aldosterone_ng_dl: float = 10.0
    adrenal_function_percent: float = 100.0
    
    # Reproductive hormones (averaged for both sexes)
    estrogen_pg_ml: float = 100.0
    testosterone_ng_dl: float = 400.0
    progesterone_ng_ml: float = 5.0
    lh_miu_ml: float = 10.0  # Luteinizing hormone
    fsh_miu_ml: float = 8.0  # Follicle stimulating hormone
    
    # Growth and metabolic hormones
    growth_hormone_ng_ml: float = 2.0
    igf1_ng_ml: float = 200.0  # Insulin-like growth factor
    
    # Metabolic effects
    metabolic_rate_percent: float = 100.0
    thermogenesis_percent: float = 100.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal endocrine state."""
        return {
            "tsh_mu_ml": 2.0,
            "t4_ug_dl": 8.0,
            "t3_ng_dl": 120.0,
            "thyroid_function_percent": 100.0,
            "cortisol_ug_dl": 12.0,
            "acth_pg_ml": 25.0,
            "aldosterone_ng_dl": 10.0,
            "adrenal_function_percent": 100.0,
            "estrogen_pg_ml": 100.0,
            "testosterone_ng_dl": 400.0,
            "progesterone_ng_ml": 5.0,
            "lh_miu_ml": 10.0,
            "fsh_miu_ml": 8.0,
            "growth_hormone_ng_ml": 2.0,
            "igf1_ng_ml": 200.0,
            "metabolic_rate_percent": 100.0,
            "thermogenesis_percent": 100.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all endocrine state variables."""
        return [
            "tsh_mu_ml", "t4_ug_dl", "t3_ng_dl", "thyroid_function_percent",
            "cortisol_ug_dl", "acth_pg_ml", "aldosterone_ng_dl",
            "adrenal_function_percent", "estrogen_pg_ml", "testosterone_ng_dl",
            "progesterone_ng_ml", "lh_miu_ml", "fsh_miu_ml",
            "growth_hormone_ng_ml", "igf1_ng_ml",
            "metabolic_rate_percent", "thermogenesis_percent"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_hypothyroidism(self, severity: str = "moderate"):
        """Transition to hypothyroidism."""
        severity_map = {"mild": 0.7, "moderate": 0.5, "severe": 0.3}
        function_factor = severity_map.get(severity, 0.5)
        
        self.thyroid_function_percent = 100.0 * function_factor
        self.t4_ug_dl = 8.0 * function_factor
        self.t3_ng_dl = 120.0 * function_factor
        self.tsh_mu_ml = 2.0 + ((1 - function_factor) * 18.0)  # Compensatory rise
        
        # Metabolic effects
        self.metabolic_rate_percent = 100.0 * function_factor
        self.thermogenesis_percent = 100.0 * function_factor
        
        self.state_classification = "HYPOTHYROID"
    
    def transition_to_cushings(self):
        """Transition to Cushing's syndrome (excess cortisol)."""
        self.cortisol_ug_dl = 40.0  # ↑233%
        self.acth_pg_ml = 5.0  # ↓80% (if pituitary suppression)
        
        # Metabolic effects
        self.metabolic_rate_percent = 120.0
        
        self.state_classification = "HYPERCORTISOLISM"
    
    def transition_to_pcos(self):
        """Transition to polycystic ovary syndrome."""
        self.testosterone_ng_dl = 800.0  # ↑100%
        self.lh_miu_ml = 20.0  # ↑100%
        self.fsh_miu_ml = 4.0  # ↓50%
        
        self.state_classification = "PCOS"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Thyroid affects metabolism
        if self.thyroid_function_percent < 70:
            effects["Heart"] = {
                "heart_rate_bpm": -(100 - self.thyroid_function_percent) / 2.0
            }
            effects["Muscles"] = {
                "metabolic_rate_percent": -(100 - self.thyroid_function_percent)
            }
        
        # Cortisol affects multiple systems
        if self.cortisol_ug_dl > 25:
            excess = (self.cortisol_ug_dl - 12) / 28.0
            effects["Immune"] = {
                "immune_suppression": excess
            }
            effects["Bones"] = {
                "bone_resorption": excess * 20.0
            }
        
        return effects
