"""
Brain (Neurological) State Machine.

Models complete neurological physiology including:
- Perfusion and metabolism
- Neurotransmitter systems
- Neurotrophic factors
- Cognitive function
- Disease states (Alzheimer's, Parkinson's, stroke)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class BrainState(OrganState):
    """Neurological state machine."""
    
    organ_name: str = "Brain"
    
    # Structural parameters
    mass_g: float = 1400.0
    neuron_count_billion: float = 86.0
    synapse_count_trillion: float = 100.0
    white_matter_volume_ml: float = 500.0
    gray_matter_volume_ml: float = 600.0
    
    # Perfusion and metabolism
    cerebral_blood_flow_ml_min: float = 750.0
    oxygen_consumption_ml_min: float = 50.0
    glucose_consumption_mg_min: float = 120.0
    
    # Neurotransmitters (nmol/g or umol/g)
    dopamine_nmol_g: float = 1.0
    serotonin_nmol_g: float = 0.5
    acetylcholine_nmol_g: float = 0.3
    gaba_umol_g: float = 2.0
    glutamate_umol_g: float = 10.0
    norepinephrine_nmol_g: float = 0.4
    
    # Neurotrophic factors
    bdnf_ng_ml: float = 25.0  # Brain-derived neurotrophic factor
    ngf_pg_ml: float = 50.0  # Nerve growth factor
    neuroplasticity_index: float = 100.0
    
    # Metabolic
    atp_production_percent: float = 100.0
    mitochondrial_function_percent: float = 100.0
    oxidative_stress_index: float = 1.0
    
    # Cognitive function
    memory_score: float = 100.0
    processing_speed_ms: float = 200.0
    executive_function_score: float = 100.0
    consciousness_level: str = "ALERT"  # ALERT, DROWSY, CONFUSED, UNCONSCIOUS
    
    # Additional parameters
    consciousness_impairment: float = 0.0  # 0-1 scale
    ammonia_toxicity: float = 0.0  # 0-1 scale
    cerebral_vasodilation: float = 0.0  # 0-1 scale
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal neurological state."""
        return {
            "mass_g": 1400.0,
            "neuron_count_billion": 86.0,
            "synapse_count_trillion": 100.0,
            "white_matter_volume_ml": 500.0,
            "gray_matter_volume_ml": 600.0,
            "cerebral_blood_flow_ml_min": 750.0,
            "oxygen_consumption_ml_min": 50.0,
            "glucose_consumption_mg_min": 120.0,
            "dopamine_nmol_g": 1.0,
            "serotonin_nmol_g": 0.5,
            "acetylcholine_nmol_g": 0.3,
            "gaba_umol_g": 2.0,
            "glutamate_umol_g": 10.0,
            "norepinephrine_nmol_g": 0.4,
            "bdnf_ng_ml": 25.0,
            "ngf_pg_ml": 50.0,
            "neuroplasticity_index": 100.0,
            "atp_production_percent": 100.0,
            "mitochondrial_function_percent": 100.0,
            "oxidative_stress_index": 1.0,
            "memory_score": 100.0,
            "processing_speed_ms": 200.0,
            "executive_function_score": 100.0,
            "consciousness_level": "ALERT",
            "consciousness_impairment": 0.0,
            "ammonia_toxicity": 0.0,
            "cerebral_vasodilation": 0.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all neurological state variables."""
        return [
            "mass_g", "neuron_count_billion", "synapse_count_trillion",
            "white_matter_volume_ml", "gray_matter_volume_ml",
            "cerebral_blood_flow_ml_min", "oxygen_consumption_ml_min",
            "glucose_consumption_mg_min", "dopamine_nmol_g", "serotonin_nmol_g",
            "acetylcholine_nmol_g", "gaba_umol_g", "glutamate_umol_g",
            "norepinephrine_nmol_g", "bdnf_ng_ml", "ngf_pg_ml",
            "neuroplasticity_index", "atp_production_percent",
            "mitochondrial_function_percent", "oxidative_stress_index",
            "memory_score", "processing_speed_ms", "executive_function_score",
            "consciousness_impairment", "ammonia_toxicity", "cerebral_vasodilation"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_alzheimers(self, stage: str = "moderate"):
        """
        Transition to Alzheimer's disease.
        
        Args:
            stage: "mild", "moderate", "severe"
        """
        stage_map = {
            "mild": 0.8,
            "moderate": 0.5,
            "severe": 0.3
        }
        
        function_factor = stage_map.get(stage, 0.5)
        
        # Neurodegeneration
        self.neuron_count_billion = 86.0 * function_factor
        self.synapse_count_trillion = 100.0 * (function_factor ** 1.5)
        
        # Neurotransmitter deficits
        self.acetylcholine_nmol_g = 0.3 * function_factor
        self.bdnf_ng_ml = 25.0 * function_factor
        
        # Metabolic dysfunction
        self.mitochondrial_function_percent = 100.0 * function_factor
        self.atp_production_percent = 100.0 * function_factor
        self.oxidative_stress_index = 1.0 + ((1 - function_factor) * 3.0)
        
        # Cognitive decline
        self.memory_score = 100.0 * function_factor
        self.processing_speed_ms = 200.0 + ((1 - function_factor) * 300.0)
        self.executive_function_score = 100.0 * function_factor
        
        self.state_classification = "DEMENTIA"
    
    def transition_to_parkinsons(self, hoehn_yahr_stage: int = 2):
        """
        Transition to Parkinson's disease.
        
        Args:
            hoehn_yahr_stage: Stage 1-5
        """
        dopamine_loss = hoehn_yahr_stage * 0.15
        
        # Dopaminergic neuron loss
        self.dopamine_nmol_g = 1.0 * (1 - dopamine_loss)
        
        # Compensatory changes
        self.norepinephrine_nmol_g = 0.4 * (1 - dopamine_loss * 0.5)
        
        # Mitochondrial dysfunction
        self.mitochondrial_function_percent = 100.0 * (1 - dopamine_loss * 0.6)
        self.oxidative_stress_index = 1.0 + (dopamine_loss * 2.0)
        
        # Motor and cognitive effects
        self.processing_speed_ms = 200.0 + (dopamine_loss * 200.0)
        self.executive_function_score = 100.0 * (1 - dopamine_loss * 0.4)
        
        self.state_classification = "NEURODEGENERATIVE"
    
    def transition_to_stroke(self, affected_volume_ml: float = 50.0):
        """
        Transition to acute stroke.
        
        Args:
            affected_volume_ml: Volume of infarcted tissue (ml)
        """
        infarct_fraction = affected_volume_ml / 1100.0  # Total brain volume ~1100ml
        
        # Acute tissue loss
        self.neuron_count_billion = 86.0 * (1 - infarct_fraction)
        
        # Perfusion deficit
        self.cerebral_blood_flow_ml_min = 750.0 * (1 - infarct_fraction * 1.5)
        
        # Metabolic crisis
        self.oxygen_consumption_ml_min = 50.0 * (1 - infarct_fraction)
        self.atp_production_percent = 100.0 * (1 - infarct_fraction * 2.0)
        
        # Functional impairment
        if affected_volume_ml > 100:
            self.consciousness_level = "UNCONSCIOUS"
            self.consciousness_impairment = 1.0
        elif affected_volume_ml > 50:
            self.consciousness_level = "CONFUSED"
            self.consciousness_impairment = 0.5
        
        self.state_classification = "STRESSED"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Autonomic control of heart
        if self.consciousness_level == "ALERT":
            effects["Heart"] = {
                "heart_rate_variability": self.neuroplasticity_index / 100.0
            }
        
        # Hypothalamic-pituitary-adrenal axis
        if self.oxidative_stress_index > 2.0:
            effects["Endocrine"] = {
                "cortisol_level": (self.oxidative_stress_index - 1.0) * 10.0
            }
        
        return effects
    
    def receive_from_heart(self, cardiac_output_l_min: float):
        """
        Receive blood flow from heart.
        
        Args:
            cardiac_output_l_min: Cardiac output
        """
        # Brain receives ~15% of cardiac output
        self.cerebral_blood_flow_ml_min = cardiac_output_l_min * 150.0
        
        # Hypoperfusion affects consciousness
        if self.cerebral_blood_flow_ml_min < 500:
            perfusion_deficit = (750 - self.cerebral_blood_flow_ml_min) / 750.0
            self.consciousness_impairment = perfusion_deficit
            
            if perfusion_deficit > 0.5:
                self.consciousness_level = "CONFUSED"
            if perfusion_deficit > 0.7:
                self.consciousness_level = "UNCONSCIOUS"
    
    def receive_from_gut(self, serotonin_production_percent: float):
        """
        Receive gut-brain axis signals.
        
        Args:
            serotonin_production_percent: Gut serotonin production (%)
        """
        # Gut produces 90% of body's serotonin
        self.serotonin_nmol_g = 0.5 * (serotonin_production_percent / 100.0)
