"""
Bone (Skeletal) State Machine.

Models skeletal physiology including:
- Bone mineral density
- Remodeling (osteoblasts/osteoclasts)
- Bone marrow (hematopoiesis)
- Disease states (osteoporosis, osteomalacia, Paget's)
"""

from dataclasses import dataclass
from typing import Dict, List, Any
from .base_organ import OrganState


@dataclass
class BoneState(OrganState):
    """Bone state machine."""
    
    organ_name: str = "Bones"
    
    # Structural
    bone_mineral_density_g_cm2: float = 1.2
    bone_mineral_density_percent: float = 100.0
    total_bone_mass_kg: float = 5.0
    
    # Remodeling
    osteoblast_activity_percent: float = 100.0  # Bone formation
    osteoclast_activity_percent: float = 100.0  # Bone resorption
    bone_formation_rate_percent: float = 100.0
    bone_resorption_rate_percent: float = 100.0
    remodeling_balance: float = 0.0  # Positive = net formation
    
    # Bone marrow
    hematopoietic_activity_percent: float = 100.0
    red_blood_cell_production_percent: float = 100.0
    white_blood_cell_production_percent: float = 100.0
    
    # Biochemical markers
    calcium_mg_dl: float = 9.5
    phosphate_mg_dl: float = 3.5
    alkaline_phosphatase_u_l: float = 70.0
    vitamin_d_ng_ml: float = 30.0
    parathyroid_hormone_pg_ml: float = 40.0
    
    # Structural integrity
    fracture_risk_percent: float = 5.0
    bone_strength_percent: float = 100.0
    
    def get_basal_state(self) -> Dict[str, Any]:
        """Return normal bone state."""
        return {
            "bone_mineral_density_g_cm2": 1.2,
            "bone_mineral_density_percent": 100.0,
            "total_bone_mass_kg": 5.0,
            "osteoblast_activity_percent": 100.0,
            "osteoclast_activity_percent": 100.0,
            "bone_formation_rate_percent": 100.0,
            "bone_resorption_rate_percent": 100.0,
            "remodeling_balance": 0.0,
            "hematopoietic_activity_percent": 100.0,
            "red_blood_cell_production_percent": 100.0,
            "white_blood_cell_production_percent": 100.0,
            "calcium_mg_dl": 9.5,
            "phosphate_mg_dl": 3.5,
            "alkaline_phosphatase_u_l": 70.0,
            "vitamin_d_ng_ml": 30.0,
            "parathyroid_hormone_pg_ml": 40.0,
            "fracture_risk_percent": 5.0,
            "bone_strength_percent": 100.0,
            "state_classification": "NORMAL"
        }
    
    def get_state_variables(self) -> List[str]:
        """Return list of all bone state variables."""
        return [
            "bone_mineral_density_g_cm2", "bone_mineral_density_percent",
            "total_bone_mass_kg", "osteoblast_activity_percent",
            "osteoclast_activity_percent", "bone_formation_rate_percent",
            "bone_resorption_rate_percent", "remodeling_balance",
            "hematopoietic_activity_percent", "red_blood_cell_production_percent",
            "white_blood_cell_production_percent", "calcium_mg_dl",
            "phosphate_mg_dl", "alkaline_phosphatase_u_l",
            "vitamin_d_ng_ml", "parathyroid_hormone_pg_ml",
            "fracture_risk_percent", "bone_strength_percent"
        ]
    
    # ==================== DISEASE TRANSITIONS ====================
    
    def transition_to_osteoporosis(self, t_score: float = -2.5):
        """Transition to osteoporosis (T-score < -2.5)."""
        # BMD reduction
        bmd_reduction = abs(t_score) / 2.5
        self.bone_mineral_density_percent = 100.0 * (1 - bmd_reduction * 0.3)
        self.bone_mineral_density_g_cm2 = 1.2 * (1 - bmd_reduction * 0.3)
        
        # Imbalanced remodeling
        self.osteoclast_activity_percent = 150.0  # ↑50%
        self.osteoblast_activity_percent = 70.0  # ↓30%
        self.bone_resorption_rate_percent = 150.0
        self.bone_formation_rate_percent = 70.0
        self.remodeling_balance = -80.0  # Net loss
        
        # Increased fracture risk
        self.fracture_risk_percent = 5.0 + (bmd_reduction * 45.0)
        self.bone_strength_percent = 100.0 * (1 - bmd_reduction * 0.4)
        
        self.state_classification = "OSTEOPOROTIC"
    
    def transition_to_osteomalacia(self):
        """Transition to osteomalacia (vitamin D deficiency)."""
        self.vitamin_d_ng_ml = 10.0  # ↓67%
        self.calcium_mg_dl = 8.0  # ↓16%
        self.phosphate_mg_dl = 2.0  # ↓43%
        self.parathyroid_hormone_pg_ml = 80.0  # ↑100% (secondary hyperparathyroidism)
        
        # Impaired mineralization
        self.bone_formation_rate_percent = 50.0
        self.bone_strength_percent = 60.0
        
        self.state_classification = "OSTEOMALACIC"
    
    def transition_to_pagets_disease(self):
        """Transition to Paget's disease (excessive remodeling)."""
        self.osteoblast_activity_percent = 300.0  # ↑200%
        self.osteoclast_activity_percent = 300.0  # ↑200%
        self.bone_formation_rate_percent = 300.0
        self.bone_resorption_rate_percent = 300.0
        
        # Disorganized bone
        self.alkaline_phosphatase_u_l = 500.0  # ↑614%
        self.bone_strength_percent = 70.0  # Weaker despite increased mass
        
        self.state_classification = "PAGETS"
    
    # ==================== INTER-ORGAN EFFECTS ====================
    
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """Return effects on other organs."""
        effects = {}
        
        # Bone marrow affects blood
        if self.hematopoietic_activity_percent < 70:
            effects["Blood"] = {
                "rbc_production": -(100 - self.hematopoietic_activity_percent),
                "wbc_production": -(100 - self.hematopoietic_activity_percent)
            }
        
        return effects
    
    def receive_from_kidneys(self, vitamin_d_activation: float):
        """Receive activated vitamin D from kidneys."""
        self.vitamin_d_ng_ml = 30.0 * (vitamin_d_activation / 100.0)
        
        # Vitamin D affects bone formation
        if vitamin_d_activation < 60:
            self.bone_formation_rate_percent = vitamin_d_activation
