"""
Base class for all organ state machines.

Provides common interface for:
- State management
- Homeostasis calculation
- Inter-organ effects
- Disease transitions
"""

from abc import ABC, abstractmethod
from typing import Dict, List, Any, Optional
from dataclasses import dataclass, field
import copy


@dataclass
class OrganState(ABC):
    """Abstract base class for all organ states."""
    
    # Common metadata
    organ_name: str = ""
    state_classification: str = "NORMAL"  # NORMAL | STRESSED | COMPENSATED | FAILING
    timestamp: float = 0.0
    
    def __post_init__(self):
        """Initialize organ to basal state."""
        self.reset_to_basal()
    
    @abstractmethod
    def get_basal_state(self) -> Dict[str, Any]:
        """
        Return normal physiological state.
        
        Returns:
            Dictionary of parameter_name -> basal_value
        """
        pass
    
    @abstractmethod
    def get_state_variables(self) -> List[str]:
        """
        Return list of all state variable names.
        
        Returns:
            List of parameter names that define this organ's state
        """
        pass
    
    def reset_to_basal(self):
        """Reset organ to basal (normal) state."""
        basal = self.get_basal_state()
        for param, value in basal.items():
            setattr(self, param, value)
        self.state_classification = "NORMAL"
    
    def get_current_state(self) -> Dict[str, Any]:
        """
        Get current state as dictionary.
        
        Returns:
            Dictionary of all current state variables
        """
        state = {}
        for var in self.get_state_variables():
            if hasattr(self, var):
                state[var] = getattr(self, var)
        return state
    
    def calculate_homeostasis_score(self) -> float:
        """
        Calculate how close current state is to basal state.
        
        Returns:
            Score from 0-100 (100 = perfect homeostasis)
        """
        basal = self.get_basal_state()
        current = self.get_current_state()
        
        deviations = []
        for param in self.get_state_variables():
            if param in basal and param in current:
                basal_val = basal[param]
                current_val = current[param]
                
                # Skip non-numeric values
                if not isinstance(basal_val, (int, float)) or not isinstance(current_val, (int, float)):
                    continue
                
                # Calculate percentage deviation
                if basal_val != 0:
                    deviation = abs((current_val - basal_val) / basal_val)
                else:
                    # For parameters where basal is 0 (e.g., edema, inflammation),
                    # use absolute value normalized to 1.0 (or a reasonable scale)
                    deviation = abs(current_val)
                
                deviations.append(deviation)
        
        if not deviations:
            return 100.0
        
        # Average deviation, convert to score. 
        # Cap individual deviations at 1.0 to prevent one massive error from zeroing the score.
        normalized_deviations = [min(1.0, d) for d in deviations]
        avg_deviation = sum(normalized_deviations) / len(normalized_deviations)
        score = max(0, 100 - (avg_deviation * 100))
        return score
    
    @abstractmethod
    def get_affected_organs(self) -> Dict[str, Dict[str, float]]:
        """
        Return effects this organ has on other organs.
        
        Returns:
            Dictionary mapping organ_name -> {parameter: delta_value}
            Example: {"kidneys": {"renal_blood_flow_ml_min": -30}}
        """
        pass
    
    def apply_effects(self, effects: Dict[str, float]):
        """
        Apply effects from other organs to this organ's state.
        
        Args:
            effects: Dictionary of parameter_name -> delta_value
        """
        for param, delta in effects.items():
            if hasattr(self, param):
                current_value = getattr(self, param)
                if isinstance(current_value, (int, float)):
                    new_value = current_value + delta
                    setattr(self, param, new_value)
    
    def get_state_summary(self) -> str:
        """
        Get human-readable summary of organ state.
        
        Returns:
            String summary
        """
        score = self.calculate_homeostasis_score()
        return f"{self.organ_name}: {self.state_classification} (Homeostasis: {score:.1f}%)"
    
    def is_healthy(self, threshold: float = 70.0) -> bool:
        """
        Check if organ is in healthy state.
        
        Args:
            threshold: Minimum homeostasis score for healthy (default 70%)
            
        Returns:
            True if homeostasis score >= threshold
        """
        return self.calculate_homeostasis_score() >= threshold
    
    def clone(self) -> 'OrganState':
        """
        Create deep copy of organ state.
        
        Returns:
            New instance with same state
        """
        return copy.deepcopy(self)
