"""
Longevity Orchestrator: Century-scale physiological time-travel.
"""

from body_state_model import HumanBodyState, BodySystemOrchestrator
from organs.geriatrics import GeriatricSpecialist
from genetics_module import GenomicsEngine
from typing import List, Dict

class LongevityOrchestrator:
    """Orchestrates multi-decade simulations with genetic modulation."""
    
    def __init__(self, body: HumanBodyState):
        self.body = body
        self.orchestrator = BodySystemOrchestrator(body)
        self.genetics = GenomicsEngine()
        self.history = []

    def simulate_lifespan(self, years: int, genetic_profile: List[str], verbose: bool = True):
        """Simulate a lifespan year-by-year."""
        print(f"\n[LONGEVITY] Initiating lifespan simulation: {years} years.")
        print(f"[LONGEVITY] Genetic Profile: {', '.join(genetic_profile) if genetic_profile else 'Baseline'}")
        
        for year in range(1, years + 1):
            age = year
            
            # 1. Apply aging with genetic modulation
            # We calculate a per-organ senescence multiplier based on genetics
            # For simplicity in this demo, we use a weighted systemic average for the specialist hook
            global_multiplier = self.genetics.calculate_integrated_multiplier(genetic_profile, "systemic")
            
            # Apply senescence
            GeriatricSpecialist.apply_senescence(
                self.body, 
                age_years=float(age), 
                senescence_rate_multiplier=global_multiplier
            )
            
            # 2. Stabilize physiological state (run short daily/weekly equivalent simulation)
            self.orchestrator.run_simulation(duration_seconds=1.0) # Mock stabilization
            
            # 3. Record snapshot
            score = self.body.get_overall_homeostasis()["overall_homeostasis_score"]
            self.history.append({"age": age, "homeostasis": score})
            
            if verbose and age % 10 == 0:
                print(f"  Age {age:3} | Homeostasis: {score:5.1f}% | Multiplier: {global_multiplier:.3f}")
                
            if score < 40.0:
                print(f"\n[CRITICAL] Biological System Failure reached at age {age}.")
                break
        
        return self.history
