"""
Genomics Engine: Mapping SNPs to physiological modifiers.
"""

from dataclasses import dataclass, field
from typing import Dict, List, Any
from body_state_model import HumanBodyState

@dataclass
class GeneticVariant:
    id: str
    name: str
    description: str
    modifiers: Dict[str, float] # Organ -> Reduction in senescence rate (e.g., 0.8 = 20% slower aging)

class GenomicsEngine:
    """Engine for applying genetic variants to physiological senescence."""
    
    def __init__(self):
        # Define Known Longevity Variants
        self.variants_library = {
            "FOXO3_LONGEVITY": GeneticVariant(
                id="FOXO3_LONGEVITY",
                name="FOXO3 Longevity Allele",
                description="Reduces systemic senescence and improves cellular stress resistance.",
                modifiers={
                    "systemic": 0.8, # 20% slower aging globally
                    "heart": 0.9,
                    "kidneys": 0.85
                }
            ),
            "APOE_E2": GeneticVariant(
                id="APOE_E2",
                name="APOE \u03b52 Variant",
                description="Protective against Alzheimer's and cognitive decline.",
                modifiers={
                    "brain": 0.7 # 30% slower brain aging
                }
            ),
            "CETP_VV": GeneticVariant(
                id="CETP_VV",
                name="CETP V/V Genotype",
                description="Associated with longevity and reduced cardiovascular disease.",
                modifiers={
                    "heart": 0.75,
                    "afterload": 0.8
                }
            )
        }

    def get_variant(self, variant_id: str) -> GeneticVariant:
        return self.variants_library.get(variant_id)

    def calculate_integrated_multiplier(self, profile: List[str], organ_name: str) -> float:
        """Calculate the combined senescence multiplier for a specific organ."""
        multiplier = 1.0
        for variant_id in profile:
            variant = self.get_variant(variant_id)
            if not variant:
                continue
                
            # Apply systemic modifier if it exists
            if "systemic" in variant.modifiers:
                multiplier *= variant.modifiers["systemic"]
                
            # Apply organ-specific modifier
            target = organ_name.lower()
            if target in variant.modifiers:
                multiplier *= variant.modifiers[target]
                
        return multiplier
