"""
Complete State-Based Physiological Model.

Main orchestrator for human body digital twin with:
- 12 organ systems
- Inter-organ communication
- Disease state transitions
- Drug intervention modeling
- Homeostasis calculation
"""

from dataclasses import dataclass, field
from typing import Dict, List, Any, Optional
import time

from organs import (
    HeartState,
    LungState,
    LiverState,
    KidneyState,
    BrainState,
    PancreasState,
    GutState,
    ImmuneState,
    EndocrineState,
    MuscleState,
    BoneState,
    SkinState
)


@dataclass
class HumanBodyState:
    """Complete state representation of human physiology."""
    
    # All 12 organ systems
    heart: HeartState = field(default_factory=HeartState)
    lungs: LungState = field(default_factory=LungState)
    liver: LiverState = field(default_factory=LiverState)
    kidneys: KidneyState = field(default_factory=KidneyState)
    brain: BrainState = field(default_factory=BrainState)
    pancreas: PancreasState = field(default_factory=PancreasState)
    gut: GutState = field(default_factory=GutState)
    immune_system: ImmuneState = field(default_factory=ImmuneState)
    endocrine: EndocrineState = field(default_factory=EndocrineState)
    muscles: MuscleState = field(default_factory=MuscleState)
    bones: BoneState = field(default_factory=BoneState)
    skin: SkinState = field(default_factory=SkinState)
    
    # Simulation metadata
    simulation_time: float = 0.0
    time_step_seconds: float = 0.1
    
    def get_all_organs(self) -> Dict[str, Any]:
        """Get dictionary of all organ instances."""
        return {
            "Heart": self.heart,
            "Lungs": self.lungs,
            "Liver": self.liver,
            "Kidneys": self.kidneys,
            "Brain": self.brain,
            "Pancreas": self.pancreas,
            "Gut": self.gut,
            "Immune System": self.immune_system,
            "Endocrine System": self.endocrine,
            "Muscles": self.muscles,
            "Bones": self.bones,
            "Skin": self.skin
        }
    
    def reset_to_basal(self):
        """Reset all organs to basal (healthy) state."""
        for organ in self.get_all_organs().values():
            organ.reset_to_basal()
        self.simulation_time = 0.0
    
    def get_overall_homeostasis(self) -> Dict[str, Any]:
        """
        Calculate overall body homeostasis.
        
        Returns:
            Dictionary with overall score and per-organ scores
        """
        organ_scores = {}
        for name, organ in self.get_all_organs().items():
            organ_scores[name] = organ.calculate_homeostasis_score()
        
        overall_score = sum(organ_scores.values()) / len(organ_scores)
        
        imbalanced_organs = [
            name for name, score in organ_scores.items() 
            if score < 70.0
        ]
        
        return {
            "overall_homeostasis_score": overall_score,
            "organ_scores": organ_scores,
            "imbalanced_organs": imbalanced_organs,
            "is_healthy": overall_score >= 70.0
        }
    
    def get_state_summary(self) -> str:
        """Get human-readable summary of body state."""
        homeostasis = self.get_overall_homeostasis()
        
        summary = f"=== Human Body State (t={self.simulation_time:.1f}s) ===\n"
        summary += f"Overall Homeostasis: {homeostasis['overall_homeostasis_score']:.1f}%\n"
        summary += f"Health Status: {'HEALTHY' if homeostasis['is_healthy'] else 'IMBALANCED'}\n\n"
        
        summary += "Organ Status:\n"
        for name, organ in self.get_all_organs().items():
            score = homeostasis['organ_scores'][name]
            summary += f"  {name}: {organ.state_classification} ({score:.1f}%)\n"
        
        if homeostasis['imbalanced_organs']:
            summary += f"\n⚠️  Imbalanced Organs: {', '.join(homeostasis['imbalanced_organs'])}\n"
        
        return summary


class BodySystemOrchestrator:
    """Manages inter-organ communication and state propagation."""
    
    def __init__(self, body_state: HumanBodyState):
        """
        Initialize orchestrator.
        
        Args:
            body_state: HumanBodyState instance to manage
        """
        self.body = body_state
        self.history: List[Dict[str, Any]] = []
    
    def propagate_state_changes(self):
        """
        Propagate effects between all organs.
        
        This implements the inter-organ communication network.
        """
        # Heart affects other organs
        heart_effects = self.body.heart.get_affected_organs()
        for organ_name, effects in heart_effects.items():
            if organ_name in self.body.get_all_organs():
                self.body.get_all_organs()[organ_name].apply_effects(effects)
        
        # Lungs affect other organs
        lung_effects = self.body.lungs.get_affected_organs()
        for organ_name, effects in lung_effects.items():
            if organ_name in self.body.get_all_organs():
                self.body.get_all_organs()[organ_name].apply_effects(effects)
        
        # Liver affects other organs
        liver_effects = self.body.liver.get_affected_organs()
        for organ_name, effects in liver_effects.items():
            if organ_name in self.body.get_all_organs():
                self.body.get_all_organs()[organ_name].apply_effects(effects)
        
        # Kidneys affect other organs
        kidney_effects = self.body.kidneys.get_affected_organs()
        for organ_name, effects in kidney_effects.items():
            if organ_name in self.body.get_all_organs():
                self.body.get_all_organs()[organ_name].apply_effects(effects)
        
        # Brain affects other organs
        brain_effects = self.body.brain.get_affected_organs()
        for organ_name, effects in brain_effects.items():
            if organ_name in self.body.get_all_organs():
                self.body.get_all_organs()[organ_name].apply_effects(effects)
        
        # Pancreas affects other organs
        pancreas_effects = self.body.pancreas.get_affected_organs()
        for organ_name, effects in pancreas_effects.items():
            if organ_name in self.body.get_all_organs():
                self.body.get_all_organs()[organ_name].apply_effects(effects)
        
        # Bidirectional connections
        self._update_bidirectional_connections()
    
    def _update_bidirectional_connections(self):
        """Update specific bidirectional organ connections."""
        # Heart ← Lungs (oxygen delivery)
        self.body.heart.receive_from_lungs(
            self.body.lungs.arterial_oxygen_saturation_percent
        )
        
        # Heart ← Kidneys (volume status)
        self.body.heart.receive_from_kidneys(
            self.body.kidneys.fluid_retention_ml
        )
        
        # Lungs ← Heart (blood flow)
        self.body.lungs.receive_from_heart(
            self.body.heart.cardiac_output_l_min
        )
        
        # Liver ← Heart (perfusion)
        self.body.liver.receive_from_heart(
            self.body.heart.cardiac_output_l_min
        )
        
        # Liver ← Pancreas (hormonal control)
        self.body.liver.receive_from_pancreas(
            self.body.pancreas.insulin_secretion_mu_ml,
            self.body.pancreas.glucagon_secretion_pg_ml
        )
        
        # Kidneys ← Heart (perfusion and pressure)
        self.body.kidneys.receive_from_heart(
            self.body.heart.cardiac_output_l_min,
            self.body.heart.afterload_mmhg
        )
        
        # Brain ← Heart (perfusion)
        self.body.brain.receive_from_heart(
            self.body.heart.cardiac_output_l_min
        )
        
        # Pancreas ← Gut (glucose signal)
        # TODO: Implement when gut is added
        # self.body.pancreas.receive_from_gut(blood_glucose)
    
    def step(self):
        """
        Execute one simulation time step.
        
        Updates all organ states and propagates effects.
        """
        # Propagate inter-organ effects
        self.propagate_state_changes()
        
        # Update simulation time
        self.body.simulation_time += self.body.time_step_seconds
        
        # Record history
        self._record_state()
    
    def _record_state(self):
        """Record current state to history."""
        state_snapshot = {
            "time": self.body.simulation_time,
            "homeostasis": self.body.get_overall_homeostasis(),
            "organ_states": {}
        }
        
        for name, organ in self.body.get_all_organs().items():
            state_snapshot["organ_states"][name] = {
                "classification": organ.state_classification,
                "homeostasis_score": organ.calculate_homeostasis_score()
            }
        
        self.history.append(state_snapshot)
    
    def run_simulation(self, duration_seconds: float, verbose: bool = False):
        """
        Run simulation for specified duration.
        
        Args:
            duration_seconds: How long to simulate
            verbose: Print progress updates
        """
        num_steps = int(duration_seconds / self.body.time_step_seconds)
        
        if verbose:
            print(f"Running simulation for {duration_seconds}s ({num_steps} steps)...")
        
        for i in range(num_steps):
            self.step()
            
            if verbose and (i % 100 == 0):
                print(f"  Step {i}/{num_steps} (t={self.body.simulation_time:.1f}s)")
        
        if verbose:
            print(f"Simulation complete. Final state:")
            print(self.body.get_state_summary())
    
    def get_history_summary(self) -> Dict[str, Any]:
        """
        Get summary of simulation history.
        
        Returns:
            Dictionary with time series data
        """
        if not self.history:
            return {}
        
        return {
            "duration_seconds": self.history[-1]["time"],
            "num_steps": len(self.history),
            "initial_homeostasis": self.history[0]["homeostasis"]["overall_homeostasis_score"],
            "final_homeostasis": self.history[-1]["homeostasis"]["overall_homeostasis_score"],
            "history": self.history
        }


# ==================== EXAMPLE USAGE ====================

def example_healthy_baseline():
    """Example: Simulate healthy baseline state."""
    print("=== Example 1: Healthy Baseline ===\n")
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print(body.get_state_summary())
    
    # Run for 10 seconds
    orchestrator.run_simulation(10.0, verbose=True)


def example_heart_failure_cascade():
    """Example: Simulate heart failure and observe cascade effects."""
    print("\n=== Example 2: Heart Failure Cascade ===\n")
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    print("Initial state:")
    print(body.get_state_summary())
    
    # Induce heart failure
    print("\n>>> Inducing heart failure...")
    body.heart.transition_to_heart_failure("ischemic")
    
    # Run simulation to see cascade effects
    orchestrator.run_simulation(60.0, verbose=True)
    
    print("\n>>> Cascade effects observed:")
    print(f"  - Cardiac output: {body.heart.cardiac_output_l_min:.2f} L/min (↓30%)")
    print(f"  - Kidney GFR: {body.kidneys.gfr_ml_min:.2f} mL/min")
    print(f"  - Kidney fluid retention: {body.kidneys.fluid_retention_ml:.0f} ml")
    print(f"  - Brain perfusion: {body.brain.cerebral_blood_flow_ml_min:.0f} ml/min")


def example_diabetes_progression():
    """Example: Simulate Type 2 diabetes progression."""
    print("\n=== Example 3: Type 2 Diabetes Progression ===\n")
    
    body = HumanBodyState()
    orchestrator = BodySystemOrchestrator(body)
    
    # Induce Type 2 diabetes
    print(">>> Inducing Type 2 diabetes (70% insulin resistance)...")
    body.pancreas.transition_to_type2_diabetes(insulin_resistance_percent=70.0)
    
    print(body.get_state_summary())
    
    # Run simulation
    orchestrator.run_simulation(30.0, verbose=True)
    
    print(f"\n>>> Metabolic effects:")
    print(f"  - Insulin sensitivity: {body.pancreas.insulin_sensitivity_index:.1f}%")
    print(f"  - Fasting glucose: {body.pancreas.fasting_glucose_mg_dl:.0f} mg/dL")
    print(f"  - HbA1c: {body.pancreas.hba1c_percent:.1f}%")


if __name__ == "__main__":
    # Run examples
    example_healthy_baseline()
    example_heart_failure_cascade()
    example_diabetes_progression()
